#!/usr/bin/perl -w
use strict;
use Common;

my $description = <<"EOM";

manages the \@lang_strings array and associated strings.txt files

Usage:

lang.pl dir:../.. show
lang.pl dir:../.. switch num1 num2

[dir]
The top-level folder of an FDSE installation.  Parallel to search.pl
with "searchdata" and "searchmods" as immediate children.

[show]
Scans all codes files in 'dir' and displays how many times each
lang_string is used.  Will print warnings for strings that aren't
used, or strings that are referenced as a range.

[switch]
Replaces the positions of lang_string "num1" and "num2" in all
the code files and in the English strings.txt file.  This is
used to move holes in strings.txt to the end where they can
be safely stripped, and to move public strings to the top of
the file so that the public process can abort reading of
strings.txt early on.

EOM

unless (($ARGV[0]) and ($ARGV[1])) {
	print $description;
	exit;
	}

my $basedir = '.';
if ($ARGV[0] =~ m!^dir:(.+)$!) {
	$basedir = $1;
	}
else {
	print $description;
	exit;
	}

if ($ARGV[1] eq 'switch') {
	&remap( $ARGV[2], $ARGV[3] );
	exit;
	}
if ($ARGV[1] eq 'show') {
	&show_lang_usage();
	exit;
	}


sub str_used {
	my ($p_str_used, $p_glob, @files) = @_;
	my $err_msg = '';
	Err: {
		my ($glob, $text) = ('', '');
		foreach (@files) {
			($err_msg, $text) = &ReadFile( $_ );
			next Err if ($err_msg);
			$glob .= $text;
			}
		$$p_glob .= $glob;
		while ($glob =~ m!^.*?lang_strings\[(\d+)\](.*)$!s) {
			$$p_str_used[$1]++;
			$glob = $2;
			}
		}
	return $err_msg;
	}


sub show_lang_usage {
	my $err_msg = '';
	Err: {

		my $glob = '';

		my (@pub, @semipub, @admin) = ();

		$err_msg = &str_used( \@pub, \$glob,
			"$basedir/search.pl",
			"$basedir/searchmods/common.pl",
			);
		next Err if ($err_msg);

		$err_msg = &str_used( \@semipub, \$glob,
			"$basedir/searchmods/common_parse_page.pl",
			);
		next Err if ($err_msg);

		$err_msg = &str_used( \@admin, \$glob,
			"$basedir/searchmods/common_admin.pl",
			);
		next Err if ($err_msg);

		my %holes = ();
		my $maxpub = 1;
		my $maxsemipub = 1;

		my $max = $#pub;
		$max = $#semipub if ($#semipub > $max);
		$max = $#admin if ($#admin > $max);

		my $engtext = '';
		($err_msg, $engtext) = &ReadFile("$basedir/searchdata/templates/english/strings.txt");
		next Err if ($err_msg);
		my @eng = (0,split(m!\n!s, $engtext));

		if ($max > ($#eng + 1)) {
			print "Warning: references are made to lang_strings which do not exist!\n";
			}

		my %engstr = ();
		for (1..$#eng) {
			my $lcstr = lc($eng[$_]);
			if ($engstr{$lcstr}) {
				print "Hey - english string $_ similar to $engstr{$lcstr} - poss. dupe:\n";
				print substr($eng[$_],0,60) . "\n";
				print substr($eng[$engstr{$lcstr}],0,60) . "\n";
				print "\n";
				}
			$engstr{$lcstr} = $_;
			}

		my $openp = 0;
		my $opens = 0;

		my $i = 1;
		while ($i < ($max + 2)) {
			my ($p, $s, $a) = ($pub[$i] || 0, $semipub[$i] || 0, $admin[$i] || 0);
			my $word = substr(($eng[$i] || ''), 0, 30);
			print "$i	$p	$s	$a	$word\n";
			$maxpub = $i if ($p);
			$maxsemipub = $i if ($s);
			$holes{$i} = 1 unless ($p or $s or $a);
			$openp = $i if (($p == 0) and ($openp == 0));
			$opens = $i if (($s == 0) and ($opens == 0) and ($openp));
			$i++;
			}

		print '-' x 78 . "\n";
		foreach (sort keys %holes) {
			print "Not used: $_	" . substr(($eng[$_] || ''),0,30) . "\n";
			}
		print '-' x 78 . "\n";

		print "Max pub: $maxpub\n";
		print "Max semipub: $maxsemipub\n";

		print '-' x 78 . "\n";
		print "Searching for range descriptors:\n";
		while ($glob =~ m!^.*?lang_strings\[(.*?)\](.*)$!s) {
			my ($x, $y) = ($1, $2);
			print "Range (do not switch): $x\n" if ($x !~ m!^\d+$!);
			$glob = $y;
			}

		print '-' x 78 . "\n";

		if ($openp < $maxpub) {
			print "Recommend switch $openp with $maxpub to condense public strings.\n";
			}
		else {
			print "Congrats: public strings are properly condensed to first $maxpub strings.\n";
			if ($opens < $maxsemipub) {
				print "Recommend switch $opens with $maxsemipub to condense semi-public strings.\n";
				}
			else {
				print "Congrats: semi-public strings are properly condensed to first $maxsemipub strings.\n";
				}
			}

		print '-' x 78 . "\n";
		last Err;
		}
	continue {
		print "Error: $err_msg.\n";
		}
	}


sub remap {
	my $err_msg = '';
	Err: {

		my ($a, $b) = @_;
		if ($a == $b) {
			$err_msg = "$a == $b";
			next Err;
			}

		# switch order in strings.txt
		my $file = "$basedir/searchdata/templates/english/strings.txt";
		my @current = &Lines($file);

		# remember, there's a #1 offset:
		my $temp = $current[$a-1];
		$current[$a-1] = $current[$b-1];
		$current[$b-1] = $temp;

		my $newtext = '';
		foreach (@current) {
			$newtext .= $_;
			}
		$err_msg = &WriteFile($file, $newtext);
		next Err if ($err_msg);

		my @files = &GetFiles( $basedir, '.pl' );

		foreach $file (@files) {
			my $text = '';
			($err_msg, $text) = &ReadFile( $file );
			next Err if ($err_msg);

			my $counta2b = scalar ($text =~ s!lang_strings\[$a\]!lang_strings[4000$b]!sg);
			my $countb2a = scalar ($text =~ s!lang_strings\[$b\]!lang_strings[4000$a]!sg);
			$text =~ s!lang_strings\[4000$a\]!lang_strings[$a]!sg;
			$text =~ s!lang_strings\[4000$b\]!lang_strings[$b]!sg;

			print "Replaced $counta2b a2b and $countb2a b2a in $file\n";

			$err_msg = &WriteFile( $file, $text );
			}

		last Err;
		}
	continue {
		print "Error: $err_msg\n";
		}
	return $err_msg;
	}
