#!/usr/bin/perl
#
# CHESS.PL beta 1
# Copyright (C) 1997 John Watson
# email: john@watson-net.com
#
# April 9, 1997
#
# -----About-----
# Jan-97: Got the barebones program working.  Only understood coordinate-coordinate
#         move notation (e.g. d3-e4).  Did not check for legal moves.
# Feb-97: Added code to support multiple simultaneous games.  Changed loadgame(),
#         savegame(), newgameform().  Added listgames().  Added additional text
#         datafile of games.
# 1-Mar-97: Added code to understand algebraic notation.  Also checking for legal
#           moves.  Implemented en-passant captures.
# 15-Mar-97: Finished code for 'lost right to castle'.  Now moving a rook or king
#            loses right to castle.  Implemented pawn promotion (e.g. b8/Q, fxg1/R).
#
# To do: Check for check, checkmate, stalemate, draw (50 move, 3 position repetition),
#        temporarily can't castle due to threat.
#
# 50 move draw: count moves since last pawn move.  if 50 then draw.
# 3 position repetition: store each position in an array in FEN.  After each move,
#   check for 3 duplicates.  If 3 dups then draw.
#
# The latest copy of this script and documentation can be obtained from
# http://www.watson-net.com/
#
# -----Distribution-----
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

$DEBUG = 0;

use CGI;
$q = new CGI;

init();

# Defaults
$motd = 'chess/motd.txt';
$gamelist = 'chess/chess.txt';
$gamedir = 'chess/';
$program = 'chess.pl';
$width = 32;
$height = 30;

# Global variables
@movelist = ();
@capturelist = ();

# Pieces
%blank   = ('light','chess/light.gif', 'dark','chess/dark.gif');
%wking   = ('light','chess/wkl.gif', 'dark','chess/wkd.gif');
%wqueen  = ('light','chess/wql.gif', 'dark','chess/wqd.gif');
%wrook   = ('light','chess/wrl.gif', 'dark','chess/wrd.gif');
%wbishop = ('light','chess/wbl.gif', 'dark','chess/wbd.gif');
%wknight = ('light','chess/wnl.gif', 'dark','chess/wnd.gif');
%wpawn   = ('light','chess/wpl.gif', 'dark','chess/wpd.gif');
%bking   = ('light','chess/bkl.gif', 'dark','chess/bkd.gif');
%bqueen  = ('light','chess/bql.gif', 'dark','chess/bqd.gif');
%brook   = ('light','chess/brl.gif', 'dark','chess/brd.gif');
%bbishop = ('light','chess/bbl.gif', 'dark','chess/bbd.gif');
%bknight = ('light','chess/bnl.gif', 'dark','chess/bnd.gif');
%bpawn   = ('light','chess/bpl.gif', 'dark','chess/bpd.gif');

# begin main
# get parameters
$move = $q->param('move');
$newwhite = $q->param('newwhite');
$newblack = $q->param('newblack');
$whitemail = $q->param('whitemail');
$blackmail = $q->param('blackmail');
$gamefile = $q->param('gamefile');
$password = $q->param('password');

# set takeback variable
$takeback = $q->param('takeback');
if ($takeback eq 'Takeback') { $takeback = 1; }
else { $takeback = 0; }

# set flip board variable
$flipboard = $q->param('flipboard');
$flip = $q->param('flip');

# set flip board variable
if ($flipboard eq 'Flip') {
    if ($flip) { $flip = 0; }
    else { $flip = 1; }
}

# print header
header();

# if $newwhite or $newblack parameters are given then start a new game
if ($newwhite || $newblack) {
    init();
    $whiteplayer = $newwhite;
    $blackplayer = $newblack;
    if ($blackplayer eq "") {
        print "<h3>You didn't enter a name for Black.</h3>";
        exit;
    }
    if ($whiteplayer eq "") {
        print "<h3>You didn't enter a name for White.</h3>";
        exit;
    }
    if ($password eq "") {
        print "<h3>You need to enter a password.</h3>";
        exit;
    }
    $gamefile = newgamefile();
    $gamepass = $password;
    savegame();
}

# if gamefile is not given then bring up new game form
if ($gamefile eq "") {
    newform();
    exit;
}

# load an existing game
loadgame();

# if the user enter a valid password, they are a 'player'
if ($gamepass eq $password) {
    $player = 1;
} else {
    $player = 0;
}

# if the game status is not 'open' then enter observation mode
if ($gamestatus ne 'open') { $player = 0; }

# do takeback
if ($takeback && $player) {
    pop @movelist;
    savegame();
    init();
    loadgame();
}

# if no names for white or black players then get the names
if ($whiteplayer eq '' || $blackplayer eq '') {
    newform();
    exit;
}

# if $move is legal make move and save game file
if ($move ne '' && $player) {
    $movestatus = movepiece($move);
    if ($movestatus == 0) { print "Illegal move.\n"; }
    if ($movestatus == -1) { print "Ambiguous move.\n"; }
    if ($movestatus == 1) {
        savegame();
        if ((tomove() eq 'w' && $whitemail ne '') ||
                (tomove() eq 'b' && $blackmail ne '')) {
            mailmessage();
        }
    }
}

# show the board
displayboard();

# print footer
footer();

exit;

sub header {
    print $q->header;
    print <<EOT;
<html>
<head>
<title>Chessboard</title>
</head>
<body bgcolor="#ffffff">
<h1>Chessboard</h1>
EOT
}

sub footer {
    print <<EOT;
</body>
</html>
EOT
}

sub newform {
    #print $q->header();
    
    # motd
    if (-e $motd) {
        print "<h2>Message of the 'Day'</h2>";
        open (FILE,"<$motd");
        print <FILE>;
        close(FILE);
    }
    
    # start game form
    print <<EOT;
<h2>Existing Games</h2>
<form action="$program" method="POST">
Select an existing game:
EOT
    listgames();
    print <<EOT;
<br>
Password (required to make moves): <input type="password" name="password"><br>
<input type="submit" value="Play/Observe">
</form>

<h2>Start a new game:</h2>
<form action="$program" method="POST">
White: <input name="newwhite"><br>
E-mail: <input name="whitemail"><br>
<br>
Black: <input name="newblack"><br>
E-mail: <input name="blackmail"><br>
<br>
Password (required to make moves): <input name="password"><br>
<input type="submit" value="New Game">
<input type="reset" value="Clear">
</form>
</body>
</html>
EOT
}

sub moveform {
    print "<form action=\"$program\" method=\"POST\">\n";
    print <<EOT;
<input type="hidden" name="gamefile" value="$gamefile">
<input type="hidden" name="password" value="$password">
EOT
    
    print <<EOT if $player;
<b>Your move:</b> <input name="move"><br>
<input type="submit" value="Move">
<input type="reset" value="Clear">
<input type="submit" name="takeback" value="Takeback">
EOT

    print <<EOT;
<input type="hidden" name="flip" value="$flip">
<input type="submit" name="flipboard" value="Flip">
</form>
<form action="$program" method="POST">
<input type="submit" value="Restart">
</form>
EOT
}

sub newgamefile {
    my $gamename = ();
    
    srand();
    
    CREATEGAMEFILENAME:
    $gamefile = substr($whiteplayer, 0, 2);
    $gamefile .= substr($blackplayer, 0, 2);
    $gamefile =~ s/(\ \ |\.|;|:|'|"|<|>)//ig;
    $gamefile .= int(rand(1)*10000)+1;
    $gamefile .= ".txt";

    if (-e $gamefile) { goto CREATEGAMEFILENAME; }

    open(GAME,">>$gamelist");
    print GAME "$gamefile\n";
    close(GAME);
    
    return $gamefile;
}

sub listgames {
    my ($file, $white, $black, $status, $i) = ();
    my @list = ();
    
    $i = 0;
    open(GAME,"<$gamelist");
    while (!eof(GAME)) {
        chop($file = <GAME>);
        open(GAME2,"<$gamedir$file");
        chop($status = <GAME2>);
        chop($junk = <GAME2>);
        chop($junk = <GAME2>);
        chop($white = <GAME2>);
        chop($junk = <GAME2>);
        chop($black = <GAME2>);
        close(GAME2);
        $list[$i] = "$status\t$white\t$black\t$file";
        $i++;
    }
    close(GAME);
    
    # sort descending
    @list = sort {$b cmp $a} @list;
    
    print '<select name="gamefile">';
    foreach (@list) {
        ($status, $white, $black, $file) = split(/\t/, $_);
        print "<option value=\"$file\">($status) $white v. $black</option>";
    }
    print "</select>";
}

sub loadgame {
    @movelist = ();
    @capturelist = ();
    my ($i, $line) = ();
    
    open(GAME,"<$gamedir$gamefile") || die "Couldn't open that gamefile.  Try again.";
    chop($gamestatus = <GAME>);
    chop($gamedate = <GAME>);
    chop($gamepass = <GAME>);
    chop($whiteplayer = <GAME>);
    chop($whitemail = <GAME>);
    chop($blackplayer = <GAME>);
    chop($blackmail = <GAME>);
    chop($lastmove = <GAME>);
    
    while (!eof(GAME)) {
        chop($line = <GAME>);
        movepiece($line);
    }
    
    close(GAME);
}

sub savegame {
    $lastmove = localtime();
    open(GAME, ">$gamedir$gamefile");
    print GAME "$gamestatus\n$gamedate\n$gamepass\n$whiteplayer\n$whitemail\n$blackplayer\n$blackmail\n$lastmove\n";
    if ($#movelist >= 0) {
        foreach (@movelist) { print GAME "$_\n"; }
    }
    close(GAME);
}

sub mailmessage {
    srand();
    my ($sec, $min, $hour) = localtime();
    my $random = rand(1)*10000;
    my $filename = $hour.$min.$sec.$random;
    
    open(MAIL, ">$filename");
    print MAIL "$whiteplayer v. $blackplayer\n\n";
    print MAIL "Moves: ";
    $num = 1;
    for($i=0; $i<=$#movelist; $i+=2) {
        print MAIL "$num. $movelist[$i] $movelist[$i+1] ";
        $num++;
    }
    print MAIL "\n\nLast move: $move";
    close(MAIL);

    $mailcmd = "/inetsrv/blat/blat.exe $filename -t ";
    if (tomove() eq 'w') {
        $mailcmd .= "$whitemail ";
    } else {
        $mailcmd .= "$blackmail ";
    }
    $mailcmd .= "-s \"Your move\" -f \"The Chessboard\"";
    
    system($mailcmd);
    
    `del $filename`;
}

sub tomove {
    if (($#movelist + 1)/2 == int(($#movelist + 1)/2)) {
        return 'w';
    } else {
        return 'b';
    }
}

sub movepiece {
    my $move = $_[0];
    my ($p, $s, $t, $f, $promote, $iscapture, $annotation, $turn, $dir) = ();
    my @startsquares = ();
    my @legalmoves = ();
    my ($i, $j, $ix, $jx, $sr, $sf, $tr, $tf, $temp) = ();

    # $p = piece name (p, n, b, r, q, k)
    # $s = starting square
    # $t = target square
    # $f = starting file to unambiguize moves
    # $promote = piece pawn is trying to promote to
    # $iscapte = if the move is a capture
    # $annotation = any notes attached to the move
    # $turn = 'w'hite or 'b'lack
    # $dir = direction of movement
    # @startsquares = holds results piecepos()
    # @legalmoves = holds all matching moves found to be legal
    # $i, $j, $ix, $jx, $temp = counters
    # $sr, $sf, $tr, $tf = temp rank and file
    
    # get turn
    $turn = tomove();
    
    # if an x is in the move then this is a capture
    if ($move =~ /x/i) { $iscapture = 1; }
    print "CAPTURE x\n" if $iscapture && $DEBUG == 1;

    # MOVE: RESIGN
    if ($move =~ /resign/i) {
        if ($turn eq 'w') {
            $gamestatus = '0-1';
        } else {
            $gamestatus = '1-0';
        }
        push @movelist, $move;
        return 1;
    }
        
    # MOVE: SHORT CASTLE (O-O)
    if (lc($move) eq "o-o") {
        print "castleshort\n" if $DEBUG == 1;
        if ($turn eq 'w' && $whitecastleshort) { return 0; }
        if ($turn eq 'b' && $blackcastleshort) { return 0; }
        # white
        if ($turn eq 'w' && !$whitecastleshort) {
            if ($board{61} eq 'K' && $board{62} eq ' ' &&
                    $board{63} eq ' ' && $board{64} eq 'R') {
                $board{61} = ' ';
                $board{62} = 'R';
                $board{63} = 'K';
                $board{64} = ' ';
                $whitecastleshort = 1;
            } else {
                return 0;
            }
        }
        # black
        if ($turn eq 'b' && !$blackcastleshort) {
            if ($board{5} eq 'k' && $board{6} eq ' ' &&
                    $board{7} eq ' ' && $board{8} eq 'r') {
                $board{5} = ' ';
                $board{6} = 'r';
                $board{7} = 'k';
                $board{8} = ' ';
                $blackcastleshort = 1;
            } else {
                return 0;
            }
        }
    }
    
    # MOVE: LONG CASTLE (O-O-O)
    if (lc($move) eq "o-o-o") {
        print "castlelong\n" if $DEBUG == 1;
        if ($turn eq 'w' && $whitecastlelong) { return 0; }
        if ($turn eq 'b' && $blackcastlelong) { return 0; }
        # white
        if ($turn eq 'w' && !$whitecastlelong) {
            if ($board{57} = 'R' && $board{58} eq ' ' &&
                    $board{59} eq ' ' && $board{60} eq ' ' &&
                    $board{61} eq 'K') {
                $board{57} = ' ';
                $board{58} = ' ';
                $board{59} = 'K';
                $board{60} = 'R';
                $board{61} = ' ';
                $whitecastlelong = 1;
            } else {
                return 0;
            }
        }
        # black
        if ($turn eq 'b' && !$blackcastlelong) {
            if ($board{1} = 'r' && $board{2} eq ' ' &&
                    $board{3} eq ' ' && $board{4} eq ' ' &&
                    $board{5} eq 'k') {
                $board{1} = ' ';
                $board{2} = ' ';
                $board{3} = 'k';
                $board{4} = 'r';
                $board{5} = ' ';
                $blackcastlelong = 1;
            } else {
                return 0;
            }
        }
    }
    
    # MOVE: CASTLE, SO SKIP REST
    # We wouldn't even be here if the castle wasn't successful so...
    if (lc($move) eq "o-o" || lc($move) eq "o-o-o") {
        push @movelist, uc($move);
        return 1;
    }
    
    # PAWN MOVE (starts with a|b|c|d|e|f|g|h)
    if ($move =~ /^[a-h]/) {
        print "PAWN MOVE\n" if $DEBUG == 1;
        $p = 'P';
        $f = substr($move,0,1);
        if ($iscapture) {
            # dxe4
            $t = substr($move,2,2);
            # dxe8/Q
            $promote = substr($move,5,1) if ($move =~ m@(/N|/B|/R|/Q)@);
        } else {
            # e4
            $t = substr($move,0,2);
            # e8/Q
            $promote = substr($move,3,1) if ($move =~ m@(/N|/B|/R|/Q)@);
        }
    }
    
    # KNIGHT, BISHOP, ROOK (starts with n|b|r and length > 2)
    if ($move =~ /^N/ && length($move) > 2) { $p = 'N'; }
    if ($move =~ /^B/ && length($move) > 2) { $p = 'B'; }
    if ($move =~ /^R/ && length($move) > 2) { $p = 'R'; }
    if ($move =~ /^[NBR]/ && length($move) > 2) {
        print "BISHOP, KNIGHT, ROOK MOVE\n" if $DEBUG == 1;
        if (!$iscapture && length($move) == 3) {
            print "SHORT\n" if $DEBUG == 1;
            $f = "";
            $t = substr($move,1,2);
        }
        if (length($move) == 4) {
            print "LONG OR CAPTURE\n" if $DEBUG == 1;
            $f = substr($move,1,1);
            if ($f eq "x") { $f = ""; }
            $t = substr($move,2,2);
        }
        if (length($move) == 5) {
            print "LONG CAPTURE\n" if $DEBUG == 1;
            $f = substr($move,1,1);
            $t = substr($move,3,2);
        }
    }
    
    # QUEEN, KING (starts with q|k)
    if ($move =~ /^Q/) { $p = 'Q'; }
    if ($move =~ /^K/) { $p = 'K'; }
    if ($move =~ /^[QK]/) {
        print "QUEEN, KING MOVE\n" if $DEBUG == 1;
        $f = "";
        if ($iscapture) {
            $t = substr($move,2,2);
        } else {
            $t = substr($move,1,2);
        }
    }
    
    # verify target square is on the board
    print "STARTING FILE: $f\n" if $DEBUG == 1;
    if ($t =~ /[a-h][1-8]/) {
        print "TARGET ON BOARD: $t\n" if $DEBUG == 1;
    } else {
        print "TARGET OFF BOARD: $t\n" if $DEBUG == 1;
        return 0;
    }
    
    # convert target square to square number
    $t = square(substr($t,0,1), substr($t,1,1));
    print "TARGET SQAURE IS: $t\n" if $DEBUG == 1;
    
    # FIND ALL POSSIBLE LEGAL MOVES
    if ($turn eq 'w') {
        @startsquares = piecepos(uc($p));
    } else {
        @startsquares = piecepos(lc($p));
    }
    
    if ($DEBUG == 1) {
        print "STARTING SQUARES: ";
        foreach(@startsquares) {
            print "$_ ";
        }
        print "\n";
    }
    
    # loop through starting squares to find a match
    foreach (@startsquares) {
        ($sf, $sr) = rankfile($_); # starting rank and file
        ($tf, $tr) = rankfile($t); # target rank and file
        
        # if this starting file is not the file the player specified, next
        if ($f && ($sf ne $f)) { next; }
        
        # if $targetsquare has a piece on it and this isn't a capture, last
        if ($board{$t} =~ /[pnbrqk]/i && !$iscapture) { last; }
        
        # if $targetsquare doesn't have a piece on it and this is a capture, last
        if ($board{$t} !~ /[pnbrqk]/i && $iscapture && $enpassant != $t) {
            last;
        }
        
        # if this is a pawn move...
        if ($p eq 'P') {
            # IF NOT (pawn only moving 1 square forward OR two squares forward from it's
            # starting rank OR making a capture) THEN next.
            if (!(abs($file{$sf}-$file{$tf}) == 0 && abs($sr-$tr) == 1 ||
                    abs($file{$sf}-$file{$tf}) == 0 && abs($sr-$tr) == 2 &&
                        ($sr==2 && $turn eq 'w' || $sr==7 && $turn eq 'b') ||
                    abs($file{$sf}-$file{$tf}) == 1 && abs($sr-$tr) == 1)) {
                next;
            }
            # IF (trying to promote and not moving to back rank) THEN next.
            if ($promote ne '' && ($turn eq 'w' && $tr != 8 ||
                    $turn eq 'b' && $tr != 1)) {
                next;
            }
            # IF (moving to back rank and not promoting) THEN next.
            if ($promote eq '' && ($turn eq 'w' && $tr == 8 ||
                    $turn eq 'b' && $tr == 1)) {
                next;
            }
        }
        
        # if this is a king move...
        if ($p eq 'K') {
            # IF NOT (moving 1 square) THEN next...
            if (!(abs($file{$sf}-$file{$tf}) == 0 && abs($sr-$tr) == 1 ||
                    abs($file{$sf}-$file{$tf}) == 1 && abs($sr-$tr) == 0 ||
                    abs($file{$sf}-$file{$tf}) == 1 && abs($sr-$tr) == 1)) {
                next;
            }
            # Lose right to castle
            if ($turn eq 'w') {
                $whitecastleshort = 1;
                $whitecastlelong = 1;
            } else {
                $blackcastleshort = 1;
                $blackcastlelong = 1;
            }               
        }
        
        # if this is a rook move...
        if ($p eq 'R') {
            # and moving from rook starting square then
            # lose right to castle on that side.
            if ($_ == 1) { $blackcastlelong = 1; }
            if ($_ == 8) { $blackcastleshort = 1; }
            if ($_ == 57) { $whitecastlelong = 1; }
            if ($_ == 64) { $whitecastleshort = 1; }
        }
        
        # if this is a knight move...
        if ($p eq 'N') {
            # ...and move is knight shaped...
            if (abs($file{$sf}-$file{$tf}) == 1 && abs($sr-$tr) == 2 ||
                abs($file{$sf}-$file{$tf}) == 2 && abs($sr-$tr) == 1) {
                # ...and player not trying to capture his own pieces.
                if ($turn eq 'w' && $board{$t} =~ /[pnbrqk\ ]/) {
                    push(@legalmoves, "$_,$t");
                }
                if ($turn eq 'b' && $board{$t} =~ /[PNBRQK\ ]/) {
                    push(@legalmoves, "$_,$t");
                }
            }
        } else {
            # this is not a knight move so get the direction of movement...
            $dir = movedir($_, $t);
            print "DIRECTION: $dir\n" if $DEBUG == 2;
            # ...and convert the direction of movement into incremements...
            if ($dir eq 'n')  { $ix =  0; $jx =  1; }
            if ($dir eq 's')  { $ix =  0; $jx = -1; }
            if ($dir eq 'e')  { $ix =  1; $jx =  0; }
            if ($dir eq 'w')  { $ix = -1; $jx =  0; }
            if ($dir eq 'ne') { $ix =  1; $jx =  1; }
            if ($dir eq 'nw') { $ix = -1; $jx =  1; }
            if ($dir eq 'se') { $ix =  1; $jx = -1; }
            if ($dir eq 'sw') { $ix = -1; $jx = -1; }
            
            # check valid directions for rooks and bishops and pawns
            if ($p eq 'R' && ($dir eq 'ne' || $dir eq 'nw' || $dir eq 'se' || $dir eq 'sw')) {
                next;
            }
            if ($p eq 'B' && ($dir eq 'n' || $dir eq 's' || $dir eq 'e' || $dir eq 'w')) {
                next;
            }
            if ($p eq 'P' && ($turn eq 'w' && ($dir eq 'e' || $dir eq 'w' || $dir eq 'se' || $dir eq 'sw' || $dir eq 's') ||
                              $turn eq 'b' && ($dir eq 'e' || $dir eq 'w' || $dir eq 'ne' || $dir eq 'nw' || $dir eq 'n'))) {
                next;
            }
            
            print "  ix: $ix  jx: $jx\n" if $DEBUG == 2;
            # ...init the starting file and rank...
            $i = $file{$sf};
            $j = $sr;
            $temp = 0;
            # ...loop through squares until we reach the target square or
            #    we go off the board or we hit a piece.
            while (($i != $file{$tf} || $j != $tr) && $i >= 1 && $i <= 8 && $j >= 1 && $j <= 8) {
                $i = $i + $ix;
                $j = $j + $jx;
                $temp = square($filen{$i},$j);
                print "TESTING SQUARE: $temp\n" if $DEBUG == 2;
                if ($board{square($filen{$i},$j)} =~ /[pnbrqk]/i) { last; }
            }
            # if the square reached is our target square...
            print "SQUARE REACHED: $temp\n" if $DEBUG == 1;
            if ($temp eq $t) {
                # ...and player not trying to capture his own pieces.
                if ($turn eq 'w' && $board{$t} =~ /[pnbrqk\ ]/) {
                    push(@legalmoves, "$_,$t");
                }
                if ($turn eq 'b' && $board{$t} =~ /[PNBRQK\ ]/) {
                    push(@legalmoves, "$_,$t");
                }
            }
        }
    }
    
    if ($DEBUG == 1) {
        print "LEGAL MOVES FOUND:\n";
        foreach(@legalmoves) {
            print "  $_\n";
        }
    }
    
    # Ambiguous (more than one legal move)
    if ($#legalmoves > 0) {
        return -1;
    }
    
    # ADD MOVE TO MOVELIST AND RETURN SUCCESS
    if ($#legalmoves == 0) {
        ($s, $t) = split(/,/, $legalmoves[0]);
        ($sf, $sr) = rankfile($s);
        ($tf, $tr) = rankfile($t);
        $p = $board{$s};
        
        # normal capture
        if ($board{$t} ne ' ') { push @capturelist, $board{$t}; }
        
        # enpassant capture
        if ($enpassant == $t) {
            if ($turn eq 'w') { push @capturelist, 'p'; $board{$t + 8} = ' '; }
            if ($turn eq 'b') { push @capturelist, 'P'; $board{$t - 8} = ' '; }
        }
        
        if ($turn eq 'w') { $p = uc($p); }
        else { $p = lc($p); }
        
        if ($promote ne '') {
            if ($turn eq 'w') { $p = uc($promote); }
            else { $p = lc($promote); }
        }
        
        $board{$t} = $p;
        $board{$s} = " ";
        push @movelist, $move;

        # if pawn move and moved 2 squares then set $enpassant flag, else
        # turn off $enpassant flag
        if (lc($p) eq 'p' && abs($sr-$tr) == 2) {
            if ($turn eq 'w') { $enpassant = $s - 8; }
            if ($turn eq 'b') { $enpassant = $s + 8; }
        } else {
            $enpassant = 0;
        }

        return 1;
    }
}

sub movedir {
    # return the direction given a starting and ending square:
    #
    # nw n ne
    #   \|/
    #  w-+-e
    #   /|\
    # sw s se
    #
    # yah, yah, compass directions. who asked you?
    
    my ($s, $t) = @_;
    my ($sf, $sr, $tf, $tr) = ();
    
    ($sf, $sr) = rankfile($s);
    ($tf, $tr) = rankfile($t);

    if ($tr > $sr && $file{$sf} == $file{$tf}) { return 'n'; } # north
    if ($tr < $sr && $file{$sf} == $file{$tf}) { return 's'; } # south
    if ($sr == $tr && $file{$tf} > $file{$sf}) { return 'e'; } # east
    if ($sr == $tr && $file{$tf} < $file{$sf}) { return 'w'; } # west
    if ($tr > $sr && $file{$tf} > $file{$sf}) { return 'ne'; } # northeast
    if ($tr < $sr && $file{$tf} > $file{$sf}) { return 'se'; } # southeast
    if ($tr < $sr && $file{$tf} < $file{$sf}) { return 'sw'; } # southwest
    if ($tr > $sr && $file{$tf} < $file{$sf}) { return 'nw'; } # northwest
    
    return 0; # starting and ending are the same?
}

sub piecepos {
    # return the squares a given piece is on
    my $piece = $_[0];
    my @squares = ();
    
    for($i=1;$i<=64;$i++) {
        if ($board{$i} eq $piece) { push(@squares, $i); }
    }
    
    return(@squares);
}

sub displayboard {
    my $i, $j, $k, $c, $p, $file, $rank, $num;
    
    print "<table><tr>\n";
    # display the board
    print "<td valign=top>";

    # player name at top of board    
    if ($flip) { print "<b>$whiteplayer</b>"; }
    else { print "<b>$blackplayer</b>"; }

    print "<table border=1><tr>\n";
    $i = 1; $k = 0;
    while ($i<=64) {
        if ($flip) { $j = 64 - $i + 1; }
        else { $j = $i; }
        
        $c = color($j);
        $p = $board{$j};
        ($file, $rank) = rankfile($j);
        
        print "<td width=$width><img width=$width height=$height alt=\"$p\" src=\"";
        print "$blank{$c}" if ($p eq ' ');
        print "$bking{$c}" if ($p eq 'k');
        print "$bqueen{$c}" if ($p eq 'q');
        print "$brook{$c}" if ($p eq 'r');
        print "$bbishop{$c}" if ($p eq 'b');
        print "$bknight{$c}" if ($p eq 'n');
        print "$bpawn{$c}" if ($p eq 'p');
        print "$wking{$c}" if ($p eq 'K');
        print "$wqueen{$c}" if ($p eq 'Q');
        print "$wrook{$c}" if ($p eq 'R');
        print "$wbishop{$c}" if ($p eq 'B');
        print "$wknight{$c}" if ($p eq 'N');
        print "$wpawn{$c}" if ($p eq 'P');
        print "\"></td>\n";
        
        $k++;
        
        if ($k/8 == int($k/8)) {
            print "<td width=$width>&nbsp;&nbsp;$rank&nbsp;&nbsp;</td></tr>\n<tr>";
            $k = 0;
        }
        $i++;
    }
    print "</tr>\n<tr>";

    if ($flip) {
        print "<td width=$width>h</td>\n";
        print "<td width=$width>g</td>\n";
        print "<td width=$width>f</td>\n";
        print "<td width=$width>e</td>\n";
        print "<td width=$width>d</td>\n";
        print "<td width=$width>c</td>\n";
        print "<td width=$width>b</td>\n";
        print "<td width=$width>a</td>\n";
    } else {
        print "<td width=$width>a</td>\n";
        print "<td width=$width>b</td>\n";
        print "<td width=$width>c</td>\n";
        print "<td width=$width>d</td>\n";
        print "<td width=$width>e</td>\n";
        print "<td width=$width>f</td>\n";
        print "<td width=$width>g</td>\n";
        print "<td width=$width>h</td>\n";
    }
    print "\n</tr></table>\n";

    # player name at bottom of board    
    if ($flip) { print "<b>$blackplayer</b>"; }
    else { print "<b>$whiteplayer</b>"; }

    print "</td>\n";

    # move and capture lists
    print "<td valign=top><b>\n";
    print "Game: $whiteplayer vs. $blackplayer [$gamefile]\n";
    print "[Observation mode]\n" if !$player;
    print "</b>";
    print "<p><b>Moves (Last move: $lastmove)</b><br>\n";
    print "<font size=2>";
    $num = 1;
    for($i=0; $i<=$#movelist; $i+=2) {
        print "<b>$num.</b> $movelist[$i]  $movelist[$i+1]\n";
        $num++;
    }
    print "</font></p>";

    print "<p><b>Captured Pieces</b><br>\n";
    @capturelist = sort @capturelist;
    foreach (@capturelist) {
        print "<img width=",$width/1.5," height=",$height/1.5," src=\"";
        print "$bking{'light'}" if ($_ eq 'k');
        print "$bqueen{'light'}" if ($_ eq 'q');
        print "$brook{'light'}" if ($_ eq 'r');
        print "$bbishop{'light'}" if ($_ eq 'b');
        print "$bknight{'light'}" if ($_ eq 'n');
        print "$bpawn{'light'}" if ($_ eq 'p');
        print "$wking{'light'}" if ($_ eq 'K');
        print "$wqueen{'light'}" if ($_ eq 'Q');
        print "$wrook{'light'}" if ($_ eq 'R');
        print "$wbishop{'light'}" if ($_ eq 'B');
        print "$wknight{'light'}" if ($_ eq 'N');
        print "$wpawn{'light'}" if ($_ eq 'P');
        print "\">\n";
    }
    print "</p>\n";
    
    if ($gamestatus eq 'open') {
        if (tomove() eq 'w') {
            print "<b>White to move.</b>";
        } else {
            print "<b>Black to move.</b>";
        }
    }

    # move form
    moveform();
    
    print "</td>\n";
    
    print "</tr></table>\n";
}

sub color {
    # return the color given square number
    # even ranks start with light squares
    # odd ranks start with dark squares
    my $square = $_[0];
    my ($f, $r) = rankfile($square);
    my $diff = $square - (64 - $r*8 + 1);
    my $color;
    
    if ($r/2 == int($r/2)) { $color = 2 + $diff; }
    else { $color = 1 + $diff; }
    
    if ($color/2 == int($color/2)) { return 'light'; }
    else { return 'dark'; }
}

sub square {
    # return the square number of a given rank and file
    my ($f, $r) = @_;
    return($file{$f} + (8-$r)*8);
}

sub rankfile {
    # return the rank and file of a given square number
    my $square = $_[0];
    my ($r, $f, $diff) = ();
    
    if ($square <=64) { $r = 1; }
    if ($square <=56) { $r = 2; }
    if ($square <=48) { $r = 3; }
    if ($square <=40) { $r = 4; }
    if ($square <=32) { $r = 5; }
    if ($square <=24) { $r = 6; }
    if ($square <=16) { $r = 7; }
    if ($square <= 8) { $r = 8; }
    
    $diff = 72 - $r * 8 - $square + 1;
    if ($diff == 8) { $f = 'a'; }
    if ($diff == 7) { $f = 'b'; }
    if ($diff == 6) { $f = 'c'; }
    if ($diff == 5) { $f = 'd'; }
    if ($diff == 4) { $f = 'e'; }
    if ($diff == 3) { $f = 'f'; }
    if ($diff == 2) { $f = 'g'; }
    if ($diff == 1) { $f = 'h'; }
    
    return($f, $r);
}

sub init {
    %file = ('a', '1', 'b', '2', 'c', '3', 'd', '4',
             'e', '5', 'f', '6', 'g', '7', 'h', '8');

    %filen = ('1', 'a', '2', 'b', '3', 'c', '4', 'd',
             '5', 'e', '6', 'f', '7', 'g', '8', 'h');
    
    %board = ( '1', 'r',  '2', 'n',  '3', 'b',  '4', 'q',
               '5', 'k',  '6', 'b',  '7', 'n',  '8', 'r',
               '9', 'p', '10', 'p', '11', 'p', '12', 'p',
              '13', 'p', '14', 'p', '15', 'p', '16', 'p',
              '17', ' ', '18', ' ', '19', ' ', '20', ' ',
              '21', ' ', '22', ' ', '23', ' ', '24', ' ',
              '25', ' ', '26', ' ', '27', ' ', '28', ' ',
              '29', ' ', '30', ' ', '31', ' ', '32', ' ',
              '33', ' ', '34', ' ', '35', ' ', '36', ' ',
              '37', ' ', '38', ' ', '39', ' ', '40', ' ',
              '41', ' ', '42', ' ', '43', ' ', '44', ' ',
              '45', ' ', '46', ' ', '47', ' ', '48', ' ',
              '49', 'P', '50', 'P', '51', 'P', '52', 'P',
              '53', 'P', '54', 'P', '55', 'P', '56', 'P',
              '57', 'R', '58', 'N', '59', 'B', '60', 'Q',
              '61', 'K', '62', 'B', '63', 'N', '64', 'R');

    @movelist = ();
    @capturelist = ();

    $whitecastleshort = 0;
    $whitecastlelong = 0;
    $blackcastleshort = 0;
    $blackcastlelong = 0;

    # set if enpassant capture is possible
    # 0 if enpassant capture is not possible
    $enpassant = 0;
    
    # status of game and date/time started
    $gamestatus = 'open';
    $gamedate = localtime();
}

