<?
/*
 * Name: Database Functions
 * Author: Thomas R. Pasawicz
 * Creation Date: April 18, 2001
 * Version: 1.11 (released September 16, 2001)
 * Function: Core DB functions for add, update, delete, find and format records.
 * (c) 2001 Thomas R. Pasawicz (webmaster@diamond-back.com)
 *
 */


/******************************************************************************
 * CORE DATABASE FUNCTIONS
 *****************************************************************************/

/*
 * 
 * Function: Format a Record
 *
 */

function recordformat($data) {

global $bannedwords;

$number_elements = sizeof($data);

$recorddata = "";
$linebreak = chr(13) . "\n";  # Linebreak & line feed characters.

for ($i = 0;$i < $number_elements;$i++) {

	$field = $data[$i];

	$field = str_replace("|", "&#124;", $field); # Convert pipes to HTML elements.

	$field = Chop($field); # Remove trailing white space.
	$field = str_replace("\\", "", $field); # Strip double slashes.
	$field = htmlspecialchars($field); # Convert HTML special characters.
	$field = addslashes($field); # Add slashes.

	$field = str_replace($linebreak, "<br>", $field);  # Convert line breaks to HTML elements.

	if ($i > 0) { $recorddata .= "|"; } # Add pipe delimiter after first field.
	$recorddata .= $field;

} # End For loop

$recorddata .= $linebreak; # Add linebreak/linefeed as record delimiter.

if ($bannedwords == "true") { # Check for banned words

	$filename = "bannedwords.txt";

	if ($fp = fopen($filename, 'r')) { # File opened for reading
		$lock = flock($fp, LOCK_SH);
		if ($lock) { # Lock achived
			$bannedwordlist = file($filename);
			$lock = flock($fp, LOCK_UN);
		} # End If - Lock achived
		fclose($fp);
	} # End If - File opened for reading

	$number_words = sizeof($bannedwordlist);

	for ($i = 0; $i < $number_words; $i++) { # Loop through banned words
		$targetword = trim($bannedwordlist[$i]);
		if ($targetword != "") { # Test for banned word
			$replace = str_repeat("*", strlen($targetword));
			$recorddata = eregi_replace($targetword, $replace, $recorddata);
		} # End If - Test for banned word
	} # End For - Loop through banned words

} # End If - Check for banned words


return $recorddata;

} # End of function - Format a Record





/*
 * 
 * Function: Update Database Configuration
 *
 */

function update_dbconfiguration() {

global $totalrecords, $uidcounter, $perfile, $datapath;

$filename = $datapath . 'dbconfiguration.dat';

$contents = "$totalrecords
$uidcounter
$perfile
$datapath
";

if ($fp = fopen($filename, 'w')) {  # File opened for writing
	$lock = flock($fp, LOCK_EX); 
	if ($lock) { # Lock achived
		fwrite($fp, $contents);
		flock($fp, LOCK_UN); 
	} # End If - Lock achived
	fclose($fp);
} else { print "ERROR: Database Configuration file couldn't be opened for writing.<br><br>\n\nFilename: $filename<br><br>\n\n";
} # End If - File opened for writing

} # End of function - Update Database Configuration




/*
 * 
 * Function: Load Database Configuration
 *
 */

function load_dbconfiguration() {

global $totalrecords, $uidcounter, $perfile, $datapath;

$filename = "data/dbconfiguration.dat";

if ($fp = fopen($filename, 'r')) { # File opened for reading
	$lock = flock($fp, LOCK_SH); 
	if ($lock) { # Lock achived
		$totalrecords = chop(fgets($fp, 5000));
		$uidcounter = chop(fgets($fp, 5000));
		$perfile = chop(fgets($fp, 5000));
		$datapath = chop(fgets($fp, 5000));
		$lock = flock($fp, LOCK_UN);
	} # End If - Lock achived
	fclose($fp);
} # End If - File opened for reading

} # End of function - Load Database Configuration





/*
 * 
 * Function: Add a Record
 *
 */

function addrecord($data) {

global $totalrecords, $uidcounter, $perfile, $datapath;

$totalrecords++; # Increment total number of records.

if ($totalrecords == 1) {  # Test for first record.
	$filenumber = 1; # Initalize for first record.
} else {
	$filenumber = intval(($totalrecords - 1) / $perfile) + 1; # Calculate file number.
}

$filename = $datapath . 'data_' . $filenumber . '.dat';


/* Write record to data file. */

if ($fp = fopen($filename, 'a')) { # File opened for appending
      $lock = flock($fp, LOCK_EX); 
	if ($lock) { # Lock achived
		fseek($fp, 0, SEEK_END);
		fwrite($fp, $data);
		flock($fp, LOCK_UN); 
	} # End If - Lock achived
	fclose($fp);
} # End If - File opened for appending

update_dbconfiguration(); # Update Database Configuration file.

return $totalrecords;

} # End of function - Add a Record





/*
 * 
 * Function: Load a Record
 *
 */

function loadrecord($record) {

global $totalrecords, $perfile, $datapath;


if ($record == 1) {  # Test for first record.
	$filenumber = 1; # Initalize for first record.
	$offset = 1;
} else {
	$filenumber = intval(($record - 1) / $perfile) + 1; # Calculate file number.
	$offset = $record - (($filenumber - 1) * $perfile);
}

$filename = $datapath . 'data_' . $filenumber . '.dat';


/* Open record for reading. */

if ($fp = fopen($filename, 'r')) { # File opened for reading
	$lock = flock($fp, LOCK_SH); 
	if ($lock) { # Lock achived
		for ($i = 1;$i <= $offset;$i++) { $data = fgets($fp, 10000); }
		$lock = flock($fp, LOCK_UN);
	} # End If - Lock achived
	fclose($fp);
} # End If - File opened for reading

return $data;

} # End of function - Load a Record




/*
 * 
 * Function: Load a range of records
 *
 */

function loadrange($start,$end) {

global $totalrecords, $perfile, $datapath;

if ($start == 1) {  # Test for first record.
	$start_file = 1; # Initalize for first record.
	$start_offset = 1;
} else {
	$start_file = intval(($start - 1) / $perfile) + 1; # Calculate file number.
	$start_offset = $start - (($start_file - 1) * $perfile);
} # End If - Test for first record.


if ($end == 1) { # Test for last record.
	$end_file = 1; # Initalize for last record.
	$end_offset = 1;
} else {
	$end_file = intval(($end - 1) / $perfile) + 1; # Calculate file number.
	$end_offset = $end - (($end_file - 1) * $perfile);
} # End If - Test for last record.


$dataindex = 1;

if ($start_file == $end_file) { # Load records from single file.

	$filename = $datapath . 'data_' . $start_file . '.dat';
	if ($fp = fopen($filename, 'r')) { # File opened for reading
		$lock = flock($fp, LOCK_SH); 
		if ($lock) { # Lock achived
			for ($i = 1;$i < $start_offset;$i++) { $skip = fgets($fp, 10000); } # Skip unneeded records
			for ($i = $start_offset;$i <= $end_offset;$i++) { # Load record and add to data array.
				$data[$dataindex] = fgets($fp, 10000);
				$dataindex++; # Increment data array index
			} # End For Loop - Load record and add to data array.
			$lock = flock($fp, LOCK_UN);
		} # End If - Lock achived
		fclose($fp);
	} # End If - File opened for reading

} # End If - Load records from single file.


if ($start_file == ($end_file - 1)) { # Load records from two files.

	$filename = $datapath . 'data_' . $start_file . '.dat'; # Load from first file.
	if ($fp = fopen($filename, 'r')) { # File opened for reading
		$lock = flock($fp, LOCK_SH); 
		if ($lock) { # Lock achived
			for ($i = 1;$i < $start_offset;$i++) { $skip = fgets($fp, 10000); } # Skip unneeded records
			for ($i = $start_offset;$i <= $perfile;$i++) { # Load record and add to data array.
				$data[$dataindex] = fgets($fp, 10000);
				$dataindex++; # Increment data array index
			} # End For Loop - Load record and add to data array.
			$lock = flock($fp, LOCK_UN);
		} # End If - Lock achived
		fclose($fp);
	} # End If - File opened for reading

	$filename = $datapath . 'data_' . $end_file . '.dat'; # Load from second file.
	if ($fp = fopen($filename, 'r')) { # File opened for reading
		$lock = flock($fp, LOCK_SH); 
		if ($lock) { # Lock achived
			for ($i = 1;$i <= $end_offset;$i++) { # Load record and add to data array.
				$data[$dataindex] = fgets($fp, 10000);
				$dataindex++; # Increment data array index
			} # End For Loop - Load record and add to data array.
			$lock = flock($fp, LOCK_UN);
		} # End If - Lock achived
		fclose($fp);
	} # End If - File opened for reading

} # End If - Load records from two files.


if ($start_file < ($end_file - 1)) { # Load records from span of 3 or more files.

	$filename = $datapath . 'data_' . $start_file . '.dat'; # Load from first file.
	if ($fp = fopen($filename, 'r')) { # File opened for reading
		$lock = flock($fp, LOCK_SH); 
		if ($lock) { # Lock achived
			for ($i = 1;$i < $start_offset;$i++) { $skip = fgets($fp, 10000); } # Skip unneeded records
			for ($i = $start_offset;$i <= $perfile;$i++) { # Load record and add to data array.
				$data[$dataindex] = fgets($fp, 10000);
				$dataindex++; # Increment data array index
			} # End For Loop - Load record and add to data array.
			$lock = flock($fp, LOCK_UN);
		} # End If - Lock achived
		fclose($fp);
	} # End If - File opened for reading

	for ($filenumber = $start_file + 1;$filenumber < $end_file;$filenumber++) { # Load middle (spanned) files
		$filename = $datapath . 'data_' . $filenumber . '.dat'; # Load from middle file.
		if ($fp = fopen($filename, 'r')) { # File opened for reading
			$lock = flock($fp, LOCK_SH); 
			if ($lock) { # Lock achived
				for ($i = 1;$i <= $perfile;$i++) { # Load record and add to data array.
					$data[$dataindex] = fgets($fp, 10000);
					$dataindex++; # Increment data array index
				} # End For Loop - Load record and add to data array.
				$lock = flock($fp, LOCK_UN);
			} # End If - Lock achived
			fclose($fp);
		} # End If - File opened for reading
	} # End If - Load middle (spanned) files

	$filename = $datapath . 'data_' . $end_file . '.dat'; # Load from end file.
	if ($fp = fopen($filename, 'r')) { # File opened for reading
		$lock = flock($fp, LOCK_SH); 
		if ($lock) { # Lock achived
			for ($i = 1;$i <= $end_offset;$i++) { # Load record and add to data array.
				$data[$dataindex] = fgets($fp, 10000);
				$dataindex++; # Increment data array index
			} # End For Loop - Load record and add to data array.
			$lock = flock($fp, LOCK_UN);
		} # End If - Lock achived
		fclose($fp);
	} # End If - File opened for reading

} # End If - Load records from span of 3 or more files.

return $data;

} # End of function - Load a range of records





/*
 * 
 * Function: Export Database
 *
 */

function exportdatabase() {

global $totalrecords, $datapath;

$start = 1;
$end = $totalrecords;

$records = loadrange($start, $end); # Load all records

$filename = $datapath . 'records.dat';
if ($fp = fopen($filename, 'w')) { # File opened for writing
      $lock = flock($fp, LOCK_EX);
	if ($lock) { # Lock achived
		for ($i = 1;$i <= $totalrecords;$i++) { # Loop through records
    			fwrite($fp, $records[$i]); # Write record
		} # End Loop through records
		flock($fp, LOCK_UN);
	} # End If - Lock achived
	fclose($fp);
} # End If - File opened for writing

} # End of function - Export Database





/*
 * 
 * Function: Import Database
 *
 */

function importdatabase() {

global $totalrecords, $uidcounter, $perfile, $datapath;

$filename = $datapath . 'records.dat';
if ($fp = fopen($filename, 'r')) { # File opened for reading
	$lock = flock($fp, LOCK_SH); 
	if ($lock) { # Lock achived
		$data = file($filename);
		$lock = flock($fp, LOCK_UN);
	} # End If - Lock achived
	fclose($fp);
} # End If - File opened for reading

$number_records = sizeof($data);
$addcounter = 0; # Initialize counter for number of records added.

if ($number_records >= 1 AND $data[0] != "") { # Make certain there is at least one new record.

	if ($totalrecords == 0) { $filenumber = 1; # Initalize for first record.
	} else { $filenumber = intval(($totalrecords) / $perfile) + 1; } # Calculate file number for the next record.

	$records_in_file = $totalrecords - (($filenumber - 1) * $perfile); # Count the number of records in current file, including the new one to be added.

	$filename = $datapath . 'data_' . $filenumber . '.dat';
	$fp = fopen($filename, 'a'); # Open file for appending.
	$lock = flock($fp, LOCK_EX); # Get an exclusive lock

	while ($addcounter < $number_records) { # Loop until all new records have been added. Note: Add Counter starts at 0.

		if ($records_in_file == $perfile) { # Start a new data file
			flock($fp, LOCK_UN); # Release lock
			fclose($fp); # Close the current file.
			$filenumber++; # Increment the file number counter.
			$records_in_file = 0; # Reset the records in file counter.
			$filename = $datapath . 'data_' . $filenumber . '.dat'; # Set next data file name.
			$fp = fopen($filename, 'a'); # Set pointer to next file.
			$lock = flock($fp, LOCK_EX); # Get an exclusive lock
		} # End If - Start a new data file

		fwrite($fp, $data[$addcounter]); # Write the current record to the database.
		$addcounter++; # Increment the Add Counter
		$records_in_file++; # Increment the number of records in this file.
		$totalrecords++; # Increment the total number of records.

	} # End While - Loop until all new records have been added

	flock($fp, LOCK_UN); # Release lock
	fclose($fp); # Close the current file.
	update_dbconfiguration(); # Update Database Configuration file.

} # End If - Make certain there is at least one new record.

return $number_records; # Return the number of records imported.

} # End of function - Import Database





/*
 * 
 * Function: Clear Database
 *
 */

function cleardatabase() {

global $totalrecords, $uidcounter, $perfile, $datapath;

if ($totalrecords > 0 ) { # Make certain there is at least one record.

	$totalfiles = intval(($totalrecords - 1) / $perfile) + 1; # Calculate the total number of files to be deleted

	for ($filenumber = 1;$filenumber <= $totalfiles;$filenumber++) { # Loop for each file to be deleted.
		$filename = $datapath . 'data_' . $filenumber . '.dat'; # Set next data file name.
		unlink($filename); # Delete the file.
	}
	
	$totalrecords = 0; # Reset the Total Record Counter.
	update_dbconfiguration(); # Update Database Configuration file.

} # End If - Make certain there is at least one record.

} # End of function - Clear Database






/*
 * 
 * Function: Update a Record
 *
 */

function updaterecord($record,$newdata) {

global $totalrecords, $perfile, $datapath;


if ($record == 1) {  # Test for first record.
	$filenumber = 1; # Initalize for first record.
	$offset = 1;
} else {
	$filenumber = intval(($record - 1) / $perfile) + 1; # Calculate file number.
	$offset = $record - (($filenumber - 1) * $perfile); # Calculate offset in file.
}


/* Calculate the number of records in the target file. */

if ($totalrecords == 1) { $lastrecord = 1; # If there is only one record, then it is also the last record.
} else {
	$lastpage = intval(($totalrecords - 1) / $perfile) + 1; # Calculate how many pages there are.
	if ($filenumber == $lastpage) { $lastrecord = $totalrecords - (($lastpage - 1) * $perfile); } # If last file, calculate how many records are present.
	else { $lastrecord = $perfile; } # Not the last file, therefore full number of records.
} # End If - Last record check


/* Load current file. */

$filename = $datapath . 'data_' . $filenumber . '.dat';
$dataindex = 1;

if ($fp = fopen($filename, 'r')) { # File opened for reading
	$lock = flock($fp, LOCK_SH); 
	if ($lock) { # Lock achived
		for ($i = 1;$i <= $lastrecord;$i++) { # Loop through records
			$data[$dataindex] = fgets($fp, 10000); # Load record into data array
			$dataindex++; # Increment data array index
		} # End Loop through records
		$lock = flock($fp, LOCK_UN);
	} # End If - Lock achived
	fclose($fp);
} # End If - File opened for reading


/* Save current file with new record. */

if ($fp = fopen($filename, 'w')) { # File opened for writing
      $lock = flock($fp, LOCK_EX);
	if ($lock) { # Lock achived
		for ($i = 1;$i <= $lastrecord;$i++) { # Loop through records
			if ($i == $offset) { $line = $newdata; } # Use new data
			else { $line = $data[$i]; } # Use existing data
    			fwrite($fp, $line);
		} # # End Loop through records
		flock($fp, LOCK_UN);
	} # End If - Lock achived
	fclose($fp);
} # End If - File opened for writing

} # End of function - Update a Record





/*
 * 
 * Function: Load First Record
 *
 */

function loadfirstrecord($filenumber) {

global $totalrecords, $perfile, $datapath;

$filename = $datapath . 'data_' . $filenumber . '.dat';
if ($fp = fopen($filename, 'r')) { # File opened for reading
	$lock = flock($fp, LOCK_SH); 
	if ($lock) { # Lock achived
		$data = fgets($fp, 10000);
		$lock = flock($fp, LOCK_UN);
	} # End If - Lock achived
	fclose($fp);
} # End If - File opened for reading

return $data;

} # End of function - Load First Record




/*
 * 
 * Function: Delete a Record
 *
 */

function delete($record) {

global $totalrecords, $uidcounter, $perfile, $datapath;

if ($record == 1) {  # Test for first record.
	$filenumber = 1; # Initalize for first record.
	$offset = 1;
} else {
	$filenumber = intval(($record - 1) / $perfile) + 1; # Calculate file number.
	$offset = $record - (($filenumber - 1) * $perfile); # Calculate offset in file.
}

$totalpages = intval(($totalrecords - 1) / $perfile) + 1; # Calculate how many pages there are.


if ($filenumber == $totalpages) { # This is the last file, do single page delete.

	$filename = $datapath . 'data_' . $filenumber . '.dat';

	if ($totalrecords == 1) { # This was the only record so just delete file.
		unlink($filename);
	} else {
		$lastrecord = $totalrecords - (($totalpages - 1) * $perfile); # Calculate how many records are present in last file.
		if ($fp = fopen($filename, 'r')) { # File opened for reading
			$lock = flock($fp, LOCK_SH); 
			if ($lock) { # Lock achived
				for ($i = 1;$i <= $lastrecord;$i++) { # Loop through records
					$data[$i] = fgets($fp, 10000); # Load record into data array
				} # End Loop through records
				$lock = flock($fp, LOCK_UN);
			} # End If - Lock achived
			fclose($fp);
		} # End If - File opened for reading

		if ($fp = fopen($filename, 'w')) { # File opened for writing
			$lock = flock($fp, LOCK_EX);
			if ($lock) { # Lock achived
				for ($i = 1;$i <= $lastrecord;$i++) { # Loop through records
					if ($i != $offset) { # Write record if it isn't the one being deleted
						$line = $data[$i];
   	 					fwrite($fp, $line);
					} # End If - Skip deleted record
				} # End Loop through records
				flock($fp, LOCK_UN);
			} # End If - Lock achived
			fclose($fp);
		} # End If - File opened for writing

	} # End If - Test for one record

} else { # Do delete with multifile shift.

	/* Update file with deleted record. */

	$holdrecord = loadfirstrecord($filenumber + 1); # Retreive first record from the next file.
	$filename = $datapath . 'data_' . $filenumber . '.dat';

	if ($fp = fopen($filename, 'r')) { # File opened for reading
		$lock = flock($fp, LOCK_SH); 
		if ($lock) { # Lock achived
			for ($i = 1;$i <= $perfile;$i++) { # Load record into data array
				$data[$i] = fgets($fp, 10000);
			} # End For Loop - Load record into data array
			$lock = flock($fp, LOCK_UN);
		} # End If - Lock achived
		fclose($fp);
	} # End If - File opened for reading

	if ($fp = fopen($filename, 'w')) { # File opened for writing
	      $lock = flock($fp, LOCK_EX);
		if ($lock) { # Lock achived
			for ($i = 1;$i <= $perfile;$i++) { # Loop through records
				if ($i != $offset) { # Write record if it isn't the one being deleted
					$line = $data[$i];
   	 				fwrite($fp, $line);
				} # End If - Skip deleted record
			} # End Loop through records
			fwrite($fp, $holdrecord); # Append record from next file
			flock($fp, LOCK_UN);
		} # End If - Lock achived
		fclose($fp);
	} # End If - File opened for writing


	/* Shift one record over from all remaining files. */

	$filenumber++; # Increment file number counter

	while ($filenumber < $totalpages) { # Loop through all pages before the last page

		$holdrecord = loadfirstrecord($filenumber + 1); # Retreive first record from the next file.
		$filename = $datapath . 'data_' . $filenumber . '.dat';

		if ($fp = fopen($filename, 'r')) { # File opened for reading
			$lock = flock($fp, LOCK_SH); 
			if ($lock) { # Lock achived
				for ($i = 1;$i <= $perfile;$i++) { # Loop through records
					$data[$i] = fgets($fp, 10000); # Load record into data array
				} # End Loop through records
				$lock = flock($fp, LOCK_UN);
			} # End If - Lock achived
			fclose($fp);
		} # End If - File opened for reading

		if ($fp = fopen($filename, 'w')) { # File opened for writing
			$lock = flock($fp, LOCK_EX);
			if ($lock) { # Lock achived
				for ($i = 2;$i <= $perfile;$i++) { # Loop through records. Note: Skip first record.
					$line = $data[$i];
   	 				fwrite($fp, $line);
				} # End Loop through records
				fwrite($fp, $holdrecord); # Append record from next file
				flock($fp, LOCK_UN);
			} # End If - Lock achived
			fclose($fp);
		} # End If - File opened for writing

		$filenumber++; # Increment file number counter
	} # End While - Loop through all pages before the last page


	/* Update or delete last file. */

	$filename = $datapath . 'data_' . $filenumber . '.dat';
	$lastrecord = $totalrecords - (($totalpages - 1) * $perfile); # Calculate how many records are present in last file.

	if ($lastrecord == 1) { # This was the only record so just delete file.
		unlink($filename);
	} else {

		if ($fp = fopen($filename, 'r')) { # File opened for reading
			$lock = flock($fp, LOCK_SH); 
			if ($lock) { # Lock achived
				for ($i = 1;$i <= $lastrecord;$i++) { # Loop through records
					$data[$i] = fgets($fp, 10000); # Load record into data array
				} # End Loop through records
				$lock = flock($fp, LOCK_UN);
			} # End If - Lock achived
			fclose($fp);
		} # End If - File opened for reading

		if ($fp = fopen($filename, 'w')) { # File opened for writing
			$lock = flock($fp, LOCK_EX);
			if ($lock) { # Lock achived
				for ($i = 2;$i <= $lastrecord;$i++) { # Note: Skip first record.
					$line = $data[$i];
   	 				fwrite($fp, $line);
				} # End For loop
				flock($fp, LOCK_UN);
			} # End If - Lock achived
			fclose($fp);
		} # End If - File opened for writing

	} # End If - Test for one record

} # End If - Single or multifile delete.

$totalrecords--; # Deincrement Total Records

update_dbconfiguration(); # Update Database Configuration file.

} # End of function - Delete a Record






/*
 * 
 * Function: Search for first match
 *
 */

function searchdatabase($targetdata,$targetfield) {

global $totalrecords, $perfile, $datapath;

if ($totalrecords == 0) { return "0"; } # If not records then return a 0 for "not found."

$filenumber = intval(($totalrecords - 1) / $perfile) + 1; # Set file number to the last file.
$recordnumber = $totalrecords; # Start search with the most recent record.
$found = "false";

while ($found == "false" AND $filenumber > 0) { # Search until found or all files have been searched.

	$filename = $datapath . 'data_' . $filenumber . '.dat'; # Load a data file into an array
	if ($fp = fopen($filename, 'r')) { # File opened for reading
		$lock = flock($fp, LOCK_SH); 
		if ($lock) { # Lock achived
			$filedata = file($filename);
			$lock = flock($fp, LOCK_UN);
		} # End If - Lock achived
		fclose($fp);
	} # End If - File opened for reading

	$number_records = sizeof($filedata);

	while ($found == "false" AND $number_records > 0) { # Search until found or all records have been searched.
		$number_records--; # Deincrement number of records (search from highest to lowest)
		$recorddata = explode("|", $filedata[$number_records]);
		if ($recorddata[$targetfield] == $targetdata) { $found = "true"; } # Found it!
		else { $recordnumber--; } # Deincrement record number.
	} # End While - Search until found or all records have been searched.

	$filenumber--; # Deincrement filenumber and try the next file.
	
} # End While - Search until found or all files have been searched.

if ($found == "false") { return "0"; } # Match not found.
else { return $recordnumber; } # Return record number of first match.

} # End of function - Search for first match






/*
 * 
 * Function: Search for all matches, partial matches, case insensitive
 *
 */

function searchdatabaseall($targetdata,$targetfield,$totalhits) {

global $totalrecords, $perfile, $datapath;

if ($totalrecords == 0) { return "0"; } # If not records then return a 0 for "not found."

$filenumber = intval(($totalrecords - 1) / $perfile) + 1; # Set file number to the last file.
$recordnumber = $totalrecords; # Start search with the most recent record.
$hits = 0; # Initialize hit counter.

while ($hits < $totalhits AND $filenumber > 0) { # Search until total hits has been reached or all files have been searched.

	$filename = $datapath . 'data_' . $filenumber . '.dat'; # Load a data file into an array
	if ($fp = fopen($filename, 'r')) { # File opened for reading
		$lock = flock($fp, LOCK_SH); 
		if ($lock) { # Lock achived
			$filedata = file($filename);
			$lock = flock($fp, LOCK_UN);
		} # End If - Lock achived
		fclose($fp);
	} # End If - File opened for reading

	$number_records = sizeof($filedata);

	while ($hits < $totalhits AND $number_records > 0) { # Search until total hits has been reached or all records have been searched.
		$number_records--; # Deincrement number of records (search from highest to lowest)
		$recorddata = explode("|", $filedata[$number_records]);
		if (stristr($recorddata[$targetfield], $targetdata)) { $foundrecords[$hits] = $recordnumber; $hits++; } # Found! Add to found records & increment hits.
		$recordnumber--;
	} # End While - Search until total hits has been reached or all records have been searched.

	$filenumber--; # Deincrement filenumber and try the next file.
	
} # End While - Search until total hits has been reached or all files have been searched.

if ($hits == 0) { return "0"; } # Match not found.
else { return $foundrecords; } # Return array of found record numbers.

} # End of function - Search for all matches





/*
 * 
 * Function: Database Integrity Check
 *
 */

function dbcheck() {

global $totalrecords, $perfile, $datapath;

if ($totalrecords == 0) { print "The dbconfiguration.dat file indicates there are no records.<br>\n"; return ""; }

$lastfile = intval(($totalrecords - 1) / $perfile) + 1; # Calculate total number of files.
$filenumber = 1; # Initialize file number counter for first file.
$lastrecord = $totalrecords - (($lastfile - 1) * $perfile); # Calculate how many records are present in last record.
$missingfile = "false";
$fileopenerror = "false";
$missingrecord = "false";
$missingfields = "false";
$missingrequiredfield = "false";

clearstatcache();

while ($filenumber <= $lastfile) { # Loop through all files
	$filename = $datapath . 'data_' . $filenumber . '.dat';
	if ( !file_exists($filename) ) { # Check for existance of data file.
		print "Missing data file number: $filenumber<br>\n"; $missingfile = "true"; }
	else { # Load and test file.

		if ($fp = fopen($filename, 'r')) { # File opened for reading
			$lock = flock($fp, LOCK_SH); 
			if ($lock) { # Lock achived
				$filedata = file($filename);
				$lock = flock($fp, LOCK_UN);
			} # End If - Lock achived
			fclose($fp);

			$number_records = sizeof($filedata);
			if (($filenumber != $lastfile AND $number_records != $perfile) OR ($filenumber == $lastfile AND $number_records != $lastrecord)) {
				print "Missing records in data file number: $filenumber<br>\n"; $missingrecord = "true"; }
			else { # Check fields.
				for ($recnum = 0;$recnum < $number_records;$recnum++) { # Loop through records.
					$recorddata = explode("|", $filedata[$recnum]);
					$number_fields = sizeof($recorddata);
					if ($number_fields != 23) { # Missing field(s)
						$linenum = $recnum + 1;
						$dbrecnum = ($filenumber - 1) * $perfile + $linenum;
						print "Incorrect number of fields in data file number: $filenumber, line: $linenum (Record #$dbrecnum)<br>\n";
						$missingfields = "true"; }
					else { # Check for required fields.
						if ($recorddata[0] == "") { # Missing name field
							$linenum = $recnum + 1;
							$dbrecnum = ($filenumber - 1) * $perfile + $linenum;
							print "Missing Name field in data file number: $filenumber, line: $linenum (Record #$dbrecnum)<br>\n";
							$missingrequiredfield = "true";
						} # End If - Missing name field
						if ($recorddata[1] == "") { # Missing date field
							$linenum = $recnum + 1;
							$dbrecnum = ($filenumber - 1) * $perfile + $linenum;
							print "Missing Date field in data file number: $filenumber, line: $linenum (Record #$dbrecnum)<br>\n";
							$missingrequiredfield = "true";
						} # End If - Missing date field
						if ($recorddata[8] == "") { # Missing comments field
							$linenum = $recnum + 1;
							$dbrecnum = ($filenumber - 1) * $perfile + $linenum;
							print "Missing Comment field in data file number: $filenumber, line: $linenum (Record #$dbrecnum)<br>\n";
							$missingrequiredfield = "true";
						} # End If - Missing comments field
					} # End If - Check for required fields.
				} # End For Loop - Loop through records.
			} # End If - Check fields.
		} else { # File couldn't be opened.
			print "File number: $filenumber couldn't be opened for reading.<br>\n"; $fileopenerror = "true";
		} # End If - File opened for reading
	} # End If - Load and test file.
	$filenumber++;
} # End While - Loop through all files

if ($missingfile == "false") { print "All data files are present.<br>\n"; }
if ($fileopenerror == "false") { print "All data files could be opened for reading.<br>\n"; }
if ($missingrecord == "false") { print "All records are present.<br>\n"; }
if ($missingfields == "false") { print "All fields are present.<br>\n"; }
if ($missingrequiredfield == "false") { print "All required fields are present.<br>\n"; }

} # End Function: Database Integrity Check

?>