/*
 *  Copyright (c) 2001 JavaZOOM : http://www.javazoom.net
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as published
 *   by the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
package javazoom.spi.vorbis.sampled.file;

import  com.jcraft.jorbis.*;
import  com.jcraft.jogg.*;

import  java.io.*;
import  java.net.URL;

import  javax.sound.sampled.AudioFileFormat;
import  javax.sound.sampled.AudioFormat;
import  javax.sound.sampled.AudioInputStream;
import  javax.sound.sampled.AudioSystem;
import  javax.sound.sampled.UnsupportedAudioFileException;

import  javax.sound.sampled.spi.AudioFileReader;



 /**
  * This class implements the AudioFileReader class and provides an
  * Ogg Vorbis file reader for use with the Java Sound Service Provider Interface.
  */
public class VorbisAudioFileReader extends AudioFileReader
{
   public static boolean 	DEBUG = false;
   public static final int 	BUFSIZE=4096*2;

   private SyncState 		oy = null;;
   private StreamState 		os = null;
   private Page 			og = null;
   private Packet 			op = null;
   private Info 			vi = null;
   private Comment 			vc = null;
   private DspState 		vd = null;
   private Block 			vb = null;
   private byte[] 			buffer = null;
   private int 				bytes = 0;
   private int 				format = -1;
   private int 				rate = 0;
   private int 				channels = 0;

   /**
    * Return the AudiopFileFormat from the given file.
    */
   public AudioFileFormat getAudioFileFormat(File file) throws UnsupportedAudioFileException, IOException
   {
      InputStream inputStream = new FileInputStream(file);
      try
      {
         return getAudioFileFormat(inputStream);
      } finally
      	{
         	inputStream.close();
      	}
   }

   /**
    * Return the AudioFileFormat from the given URL.
    */
   public AudioFileFormat getAudioFileFormat(URL url) throws UnsupportedAudioFileException, IOException
   {
      InputStream inputStream = url.openStream();
      try
      {
         return getAudioFileFormat(inputStream);
      } finally
      	{
         	inputStream.close();
      	}
   }

   /**
    * Return the AudioFileFormat from the given InputStream.
    */
   public AudioFileFormat getAudioFileFormat(InputStream inputStream) throws UnsupportedAudioFileException, IOException
   {
      return getAudioFileFormat(inputStream,new ByteArrayOutputStream());
   }

   /**
    * Return the AudioFileFormat from the given InputStream. Implementation.
    */
   protected AudioFileFormat getAudioFileFormat(InputStream bitStream, ByteArrayOutputStream baos) throws UnsupportedAudioFileException, IOException
   {
    if (DEBUG) System.err.println("getAudioFileFormat");
	init_jorbis();

    //loop:
	  // Obviously the jOrbis decoder is NOT OO :-)
      int eos=0;
      int index=oy.buffer(BUFSIZE);
      buffer=oy.data;
      try
      {
		  bytes=bitStream.read(buffer, index, BUFSIZE);
		  if (DEBUG) System.err.println("Read : "+buffer.length);
		  baos.write(buffer,0,buffer.length);

	  } catch(Exception e)
        {
          if (DEBUG) System.err.println(e.getMessage());
          throw new UnsupportedAudioFileException(e.getMessage());
        }
      oy.wrote(bytes);
      if(oy.pageout(og)!=1)
      {
        //if(bytes<BUFSIZE) break;
        if (DEBUG) System.err.println("Input does not appear to be an Ogg bitstream.");
        throw new UnsupportedAudioFileException("Input does not appear to be an Ogg bitstream.");
      }

      os.init(og.serialno());
      os.reset();
      vi.init();
      vc.init();
      if(os.pagein(og)<0)
      {
        // error; stream version mismatch perhaps
        if (DEBUG) System.err.println("Error reading first page of Ogg bitstream data.");
        throw new UnsupportedAudioFileException("Error reading first page of Ogg bitstream data.");
      }

      if(os.packetout(op)!=1)
      {
        // no page? must not be vorbis
        if (DEBUG) System.err.println("Error reading initial header packet.");
        throw new UnsupportedAudioFileException("Error reading initial header packet.");
        //break;
      }

      if(vi.synthesis_headerin(vc, op)<0)
      {
        // error case; not a vorbis header
        if (DEBUG) System.err.println("This Ogg bitstream does not contain Vorbis audio data.");
        throw new UnsupportedAudioFileException("This Ogg bitstream does not contain Vorbis audio data.");
      }

      int i=0;
      while(i<2)
      {
        while(i<2)
        {
	  		int result=oy.pageout(og);
	  		if(result==0) break; // Need more data
	  		if(result==1)
	  		{
            	os.pagein(og);
  	    		while(i<2)
  	    		{
	      			result=os.packetout(op);
	      			if(result==0) break;
	      			if(result==-1)
	      			{
	        			if (DEBUG) System.err.println("Corrupt secondary header.  Exiting.");
        				throw new UnsupportedAudioFileException("Corrupt secondary header.  Exiting.");
                		//return;
                		//break;// loop;
	      			}
	      			vi.synthesis_headerin(vc, op);
	      			i++;
	    		}
	  		}
	  	}
        index=oy.buffer(BUFSIZE);
        buffer=oy.data;
        try
        {
			bytes=bitStream.read(buffer, index, BUFSIZE);
			if (DEBUG) System.err.println("Read2 : "+buffer.length);
			baos.write(buffer,0,buffer.length);
		} catch(Exception e)
		  {
              if (DEBUG) System.err.println(e.getMessage());
              throw new UnsupportedAudioFileException(e.getMessage());
		  }
        if(bytes==0 && i<2)
        {
	  		if (DEBUG) System.err.println("End of file before finding all Vorbis headers!");
          	throw new UnsupportedAudioFileException("End of file before finding all Vorbis headers!");
		}
		oy.wrote(bytes);
      }

      {
        byte[][] ptr=vc.user_comments;
        for(int j=0; j<ptr.length;j++)
        {
          if(ptr[j]==null) break;
          if (DEBUG) System.err.println("Comment: "+new String(ptr[j], 0, ptr[j].length-1));
        }
        if (DEBUG) System.err.println("Bitstream is "+vi.channels+" channel, "+vi.rate+"Hz");
        if (DEBUG) System.err.println("Encoded by: "+new String(vc.vendor, 0, vc.vendor.length-1)+"\n");
      }

      AudioFormat.Encoding encoding = VorbisEncoding.VORBISENC;
      AudioFormat format = new AudioFormat( encoding, vi.rate, AudioSystem.NOT_SPECIFIED, vi.channels, AudioSystem.NOT_SPECIFIED, AudioSystem.NOT_SPECIFIED,true);
      AudioFileFormat.Type type = VorbisFileFormatType.VORBIS;
      if (encoding.equals(VorbisEncoding.VORBISENC)) type = VorbisFileFormatType.VORBIS;
      return new AudioFileFormat(type,format,AudioSystem.NOT_SPECIFIED);
   }

   /**
    * Return the AudioInputStream from the given InputStream.
    */
   public AudioInputStream getAudioInputStream( InputStream inputStream ) throws UnsupportedAudioFileException, IOException
   {
      // Save byte header since this method must return the stream opened at byte 0.
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      AudioFileFormat audioFileFormat = getAudioFileFormat(inputStream,baos);
      if (DEBUG) System.err.println("Size : "+baos.size());
      SequenceInputStream sequenceInputStream = new SequenceInputStream( new ByteArrayInputStream(baos.toByteArray()), inputStream);
      return new AudioInputStream(sequenceInputStream, audioFileFormat.getFormat(), audioFileFormat.getFrameLength());
   }

   /**
    * Return the AudioInputStream from the given File.
    */
   public AudioInputStream getAudioInputStream(File file) throws UnsupportedAudioFileException, IOException
   {
      InputStream inputStream = new FileInputStream(file);
      try
      {
         return getAudioInputStream(inputStream);
      } catch (UnsupportedAudioFileException e)
        {
         	inputStream.close();
         	throw e;
        }
        catch ( IOException e )
        {
         	inputStream.close();
         	throw e;
      	}
   }

   /**
    * Return the AudioInputStream from the given URL.
    */
   public AudioInputStream getAudioInputStream(URL url) throws UnsupportedAudioFileException, IOException
   {
      InputStream inputStream = url.openStream();
      try
      {
         return getAudioInputStream(inputStream);
      } catch (UnsupportedAudioFileException e)
      	{
         	inputStream.close();
         	throw e;
      	}
      	catch (IOException e)
      	{
         	inputStream.close();
         	throw e;
      	}
   }

  /**
   * Inits Vorbis.
   */
  private void init_jorbis()
  {
    oy=new SyncState();
    os=new StreamState();
    og=new Page();
    op=new Packet();
    vi=new Info();
    vc=new Comment();
    vd=new DspState();
    vb=new Block(vd);
    buffer=null;
    bytes=0;
    oy.init();
  }
}
