/*
 *  Copyright (c) 2001 JavaZOOM : http://www.javazoom.net
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as published
 *   by the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
package	javazoom.spi.vorbis.sampled.convert;

import	java.io.File;
import	java.io.FileOutputStream;
import	java.io.ByteArrayInputStream;
import	java.io.InputStream;
import	java.io.IOException;
import	java.io.OutputStream;
import  javax.sound.sampled.UnsupportedAudioFileException;

import	java.util.Arrays;

import	javax.sound.sampled.AudioFormat;
import	javax.sound.sampled.AudioInputStream;
import	javax.sound.sampled.spi.FormatConversionProvider;

import	org.tritonus.TDebug;
import	org.tritonus.util.*;
import	org.tritonus.sampled.convert.*;
import	javazoom.spi.vorbis.sampled.file.VorbisEncoding;


import  com.jcraft.jorbis.*;
import  com.jcraft.jogg.*;

/**
 * ConversionProvider for VORBIS files.
 */
public class VorbisFormatConversionProvider extends TMatrixFormatConversionProvider
{
	// TODO: frame size, frame rate, sample size, endianess?
	private static final AudioFormat[]	INPUT_FORMATS =
	{
		new AudioFormat(VorbisEncoding.VORBISENC, 32000.0F, -1, 1, -1, -1, false),	// 0
		new AudioFormat(VorbisEncoding.VORBISENC, 32000.0F, -1, 2, -1, -1, false),	// 1
		new AudioFormat(VorbisEncoding.VORBISENC, 44100.0F, -1, 1, -1, -1, false),	// 2
		new AudioFormat(VorbisEncoding.VORBISENC, 44100.0F, -1, 2, -1, -1, false),	// 3
		new AudioFormat(VorbisEncoding.VORBISENC, 48000.0F, -1, 1, -1, -1, false),	// 4
		new AudioFormat(VorbisEncoding.VORBISENC, 48000.0F, -1, 2, -1, -1, false),	// 5

		new AudioFormat(VorbisEncoding.VORBISENC, 16000.0F, -1, 1, -1, -1, false),	// 18
		new AudioFormat(VorbisEncoding.VORBISENC, 16000.0F, -1, 2, -1, -1, false),	// 19
		new AudioFormat(VorbisEncoding.VORBISENC, 22050.0F, -1, 1, -1, -1, false),	// 20
		new AudioFormat(VorbisEncoding.VORBISENC, 22050.0F, -1, 2, -1, -1, false),	// 21
		new AudioFormat(VorbisEncoding.VORBISENC, 24000.0F, -1, 1, -1, -1, false),	// 22
		new AudioFormat(VorbisEncoding.VORBISENC, 24000.0F, -1, 2, -1, -1, false),	// 23

		new AudioFormat(VorbisEncoding.VORBISENC, 8000.0F, -1, 1, -1, -1, false),	// 36
		new AudioFormat(VorbisEncoding.VORBISENC, 8000.0F, -1, 2, -1, -1, false),	// 37
		new AudioFormat(VorbisEncoding.VORBISENC, 11025.0F, -1, 1, -1, -1, false),	// 38
		new AudioFormat(VorbisEncoding.VORBISENC, 11025.0F, -1, 2, -1, -1, false),	// 39
		new AudioFormat(VorbisEncoding.VORBISENC, 12000.0F, -1, 1, -1, -1, false),	// 40
		new AudioFormat(VorbisEncoding.VORBISENC, 12000.0F, -1, 2, -1, -1, false),	// 41
	};

	private static final AudioFormat[]	OUTPUT_FORMATS =
	{
		new AudioFormat(8000.0F, 16, 1, true, false),	// 0
		new AudioFormat(8000.0F, 16, 1, true, true),	// 1
		new AudioFormat(8000.0F, 16, 2, true, false),	// 2
		new AudioFormat(8000.0F, 16, 2, true, true),	// 3
/*	24 and 32 bit not yet possible
		new AudioFormat(8000.0F, 24, 1, true, false),
		new AudioFormat(8000.0F, 24, 1, true, true),
		new AudioFormat(8000.0F, 24, 2, true, false),
		new AudioFormat(8000.0F, 24, 2, true, true),
		new AudioFormat(8000.0F, 32, 1, true, false),
		new AudioFormat(8000.0F, 32, 1, true, true),
		new AudioFormat(8000.0F, 32, 2, true, false),
		new AudioFormat(8000.0F, 32, 2, true, true),
*/
		new AudioFormat(11025.0F, 16, 1, true, false),	// 4
		new AudioFormat(11025.0F, 16, 1, true, true),	// 5
		new AudioFormat(11025.0F, 16, 2, true, false),	// 6
		new AudioFormat(11025.0F, 16, 2, true, true),	// 7
/*	24 and 32 bit not yet possible
		new AudioFormat(11025.0F, 24, 1, true, false),
		new AudioFormat(11025.0F, 24, 1, true, true),
		new AudioFormat(11025.0F, 24, 2, true, false),
		new AudioFormat(11025.0F, 24, 2, true, true),
		new AudioFormat(11025.0F, 32, 1, true, false),
		new AudioFormat(11025.0F, 32, 1, true, true),
		new AudioFormat(11025.0F, 32, 2, true, false),
		new AudioFormat(11025.0F, 32, 2, true, true),
*/
		new AudioFormat(12000.0F, 16, 1, true, false),	// 8
		new AudioFormat(12000.0F, 16, 1, true, true),	// 9
		new AudioFormat(12000.0F, 16, 2, true, false),	// 10
		new AudioFormat(12000.0F, 16, 2, true, true),	// 11
/*	24 and 32 bit not yet possible
		new AudioFormat(12000.0F, 24, 1, true, false),
		new AudioFormat(12000.0F, 24, 1, true, true),
		new AudioFormat(12000.0F, 24, 2, true, false),
		new AudioFormat(12000.0F, 24, 2, true, true),
		new AudioFormat(12000.0F, 32, 1, true, false),
		new AudioFormat(12000.0F, 32, 1, true, true),
		new AudioFormat(12000.0F, 32, 2, true, false),
		new AudioFormat(12000.0F, 32, 2, true, true),
*/
		new AudioFormat(16000.0F, 16, 1, true, false),	// 12
		new AudioFormat(16000.0F, 16, 1, true, true),	// 13
		new AudioFormat(16000.0F, 16, 2, true, false),	// 14
		new AudioFormat(16000.0F, 16, 2, true, true),	// 15
/*	24 and 32 bit not yet possible
		new AudioFormat(16000.0F, 24, 1, true, false),
		new AudioFormat(16000.0F, 24, 1, true, true),
		new AudioFormat(16000.0F, 24, 2, true, false),
		new AudioFormat(16000.0F, 24, 2, true, true),
		new AudioFormat(16000.0F, 32, 1, true, false),
		new AudioFormat(16000.0F, 32, 1, true, true),
		new AudioFormat(16000.0F, 32, 2, true, false),
		new AudioFormat(16000.0F, 32, 2, true, true),
*/
		new AudioFormat(22050.0F, 16, 1, true, false),	// 16
		new AudioFormat(22050.0F, 16, 1, true, true),	// 17
		new AudioFormat(22050.0F, 16, 2, true, false),	// 18
		new AudioFormat(22050.0F, 16, 2, true, true),	// 19
/*	24 and 32 bit not yet possible
		new AudioFormat(22050.0F, 24, 1, true, false),
		new AudioFormat(22050.0F, 24, 1, true, true),
		new AudioFormat(22050.0F, 24, 2, true, false),
		new AudioFormat(22050.0F, 24, 2, true, true),
		new AudioFormat(22050.0F, 32, 1, true, false),
		new AudioFormat(22050.0F, 32, 1, true, true),
		new AudioFormat(22050.0F, 32, 2, true, false),
		new AudioFormat(22050.0F, 32, 2, true, true),
*/
		new AudioFormat(24000.0F, 16, 1, true, false),	// 20
		new AudioFormat(24000.0F, 16, 1, true, true),	// 21
		new AudioFormat(24000.0F, 16, 2, true, false),	// 22
		new AudioFormat(24000.0F, 16, 2, true, true),	// 23
/*	24 and 32 bit not yet possible
		new AudioFormat(24000.0F, 24, 1, true, false),
		new AudioFormat(24000.0F, 24, 1, true, true),
		new AudioFormat(24000.0F, 24, 2, true, false),
		new AudioFormat(24000.0F, 24, 2, true, true),
		new AudioFormat(24000.0F, 32, 1, true, false),
		new AudioFormat(24000.0F, 32, 1, true, true),
		new AudioFormat(24000.0F, 32, 2, true, false),
		new AudioFormat(24000.0F, 32, 2, true, true),
*/
		new AudioFormat(32000.0F, 16, 1, true, false),	// 24
		new AudioFormat(32000.0F, 16, 1, true, true),	// 25
		new AudioFormat(32000.0F, 16, 2, true, false),	// 26
		new AudioFormat(32000.0F, 16, 2, true, true),	// 27
/*	24 and 32 bit not yet possible
		new AudioFormat(32000.0F, 24, 1, true, false),
		new AudioFormat(32000.0F, 24, 1, true, true),
		new AudioFormat(32000.0F, 24, 2, true, false),
		new AudioFormat(32000.0F, 24, 2, true, true),
		new AudioFormat(32000.0F, 32, 1, true, false),
		new AudioFormat(32000.0F, 32, 1, true, true),
		new AudioFormat(32000.0F, 32, 2, true, false),
		new AudioFormat(32000.0F, 32, 2, true, true),
*/
		new AudioFormat(44100.0F, 16, 1, true, false),	// 28
		new AudioFormat(44100.0F, 16, 1, true, true),	// 29
		new AudioFormat(44100.0F, 16, 2, true, false),	// 30
		new AudioFormat(44100.0F, 16, 2, true, true),	// 31
/*	24 and 32 bit not yet possible
		new AudioFormat(44100.0F, 24, 1, true, false),
		new AudioFormat(44100.0F, 24, 1, true, true),
		new AudioFormat(44100.0F, 24, 2, true, false),
		new AudioFormat(44100.0F, 24, 2, true, true),
		new AudioFormat(44100.0F, 32, 1, true, false),
		new AudioFormat(44100.0F, 32, 1, true, true),
		new AudioFormat(44100.0F, 32, 2, true, false),
		new AudioFormat(44100.0F, 32, 2, true, true),
*/
		new AudioFormat(48000.0F, 16, 1, true, false),	// 32
		new AudioFormat(48000.0F, 16, 1, true, true),	// 33
		new AudioFormat(48000.0F, 16, 2, true, false),	// 34
		new AudioFormat(48000.0F, 16, 2, true, true),	// 35
/*	24 and 32 bit not yet possible
		new AudioFormat(48000.0F, 24, 1, true, false),
		new AudioFormat(48000.0F, 24, 1, true, true),
		new AudioFormat(48000.0F, 24, 2, true, false),
		new AudioFormat(48000.0F, 24, 2, true, true),
		new AudioFormat(48000.0F, 32, 1, true, false),
		new AudioFormat(48000.0F, 32, 1, true, true),
		new AudioFormat(48000.0F, 32, 2, true, false),
		new AudioFormat(48000.0F, 32, 2, true, true),
*/
	};


	private static final boolean	t = true;
	private static final boolean	f = false;

	/*
	 *	One row for each source format.
	 */
	private static final boolean[][]	CONVERSIONS =
	{
		{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,t,t,f,f,f,f, f,f,f,f,f,f},	// 0
		{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,t,t,f,f, f,f,f,f,f,f},	// 1
		{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,t,t, f,f,f,f,f,f},	// 2
		{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, t,t,f,f,f,f},	// 3
		{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,t,t,f,f},	// 4
		{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,t,t},	// 5

		{f,f,f,f,f,f,f,f,f,f, f,f,t,t,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f},	// 18
		{f,f,f,f,f,f,f,f,f,f, f,f,f,f,t,t,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f},	// 19
		{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,t,t,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f},	// 20
		{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,t,t, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f},	// 21
		{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, t,t,f,f,f,f,f,f,f,f, f,f,f,f,f,f},	// 22
		{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,t,t,f,f,f,f,f,f, f,f,f,f,f,f},	// 23

		{t,t,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f},	// 36
		{f,f,t,t,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f},	// 37
		{f,f,f,f,t,t,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f},	// 38
		{f,f,f,f,f,f,t,t,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f},	// 39
		{f,f,f,f,f,f,f,f,t,t, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f},	// 40
		{f,f,f,f,f,f,f,f,f,f, t,t,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f},	// 41

	};

	/**
	 * Constructor.
	 */
	public VorbisFormatConversionProvider()
	{
		super(Arrays.asList(INPUT_FORMATS),Arrays.asList(OUTPUT_FORMATS), CONVERSIONS);
	}


	/**
	 * Returns converted AudioInputStream.
	 */
	public AudioInputStream getAudioInputStream(AudioFormat targetFormat, AudioInputStream audioInputStream)
	{
		if (isConversionSupported(targetFormat,audioInputStream.getFormat()))
		{
			return new DecodedVorbisAudioInputStream(targetFormat, audioInputStream);
		}
		else
		{
			throw new IllegalArgumentException("conversion not supported");
		}
	}

	/**
	 * This class implements the Vorbis decoding.
	 */
	public static class DecodedVorbisAudioInputStream extends TAsynchronousFilteredAudioInputStream
	{
   		public static boolean 	DEBUG = false;
   		public static final int BUFSIZE=4096*2;

		private InputStream		bitStream;

   		private SyncState 		oy = null;;
   		private StreamState 	os = null;
   		private Page 			og = null;
   		private Packet 			op = null;
   		private Info 			vi = null;
   		private Comment 		vc = null;
   		private DspState 		vd = null;
   		private Block 			vb = null;
   		private byte[] 			buffer = null;
   		private int 			bytes = 0;

  		private static int 		convsize=BUFSIZE*2;
  		private static byte[] 	convbuffer=new byte[convsize];

		private double[][][] 	_pcm = null;
		private float[][][] 	_pcmf = null;
		private int[] 			_index = null;
		private int				eos = 0;
		private int				i = 0;
		private int				index = 0;
		private int				result = 0;
		private int				loop = 1;

		/**
		 * Constructor.
		 */
		public DecodedVorbisAudioInputStream(AudioFormat outputFormat, AudioInputStream bitStream)
		{
			// TODO: try to find out length (possible?)
			super(bitStream,outputFormat, -1);
			this.bitStream = bitStream;
    		oy=new SyncState();
    		os=new StreamState();
    		og=new Page();
    		op=new Packet();
    		vi=new Info();
    		vc=new Comment();
    		vd=new DspState();
    		vb=new Block(vd);
    		buffer=null;
    		bytes=0;
    		oy.init();
			reinit();
      		convsize=BUFSIZE/vi.channels;

      		vd.synthesis_init(vi);
      		vb.init(vd);

      		_pcm=new double[1][][];
      		_pcmf=new float[1][][];
      		_index=new int[vi.channels];

      		 loop = 1;
		}

		/**
		 * Reads Header. Should be removed.
		 */
		private void reinit()
		{
	      if (DEBUG) System.err.println("REINIT");
	      int eos=0;
	      int index=oy.buffer(BUFSIZE);
	      buffer=oy.data;
	      try
	      {
			  bytes=bitStream.read(buffer, index, BUFSIZE);
		  } catch(Exception e)
	        {
	          if (DEBUG) System.err.println(e.getMessage());
	        }
	      oy.wrote(bytes);

	      if(oy.pageout(og)!=1)
	      {
	        //if(bytes<BUFSIZE) break;
	        if (DEBUG) System.err.println("Input does not appear to be an Ogg bitstream.");
	      }

	      os.init(og.serialno());
	      os.reset();
	      vi.init();
	      vc.init();
	      if(os.pagein(og)<0)
	      {
	        // error; stream version mismatch perhaps
	        if (DEBUG) System.err.println("Error reading first page of Ogg bitstream data.");
	      }

	      if(os.packetout(op)!=1)
	      {
	        // no page? must not be vorbis
	        if (DEBUG) System.err.println("Error reading initial header packet.");
	        //break;
	      }

	      if(vi.synthesis_headerin(vc, op)<0)
	      {
	        // error case; not a vorbis header
	        if (DEBUG) System.err.println("This Ogg bitstream does not contain Vorbis audio data.");
	      }

	      int i=0;
	      while(i<2)
	      {
	        while(i<2)
	        {
		  		int result=oy.pageout(og);
		  		if(result==0) break; // Need more data
		  		if(result==1)
		  		{
	            	os.pagein(og);
	  	    		while(i<2)
	  	    		{
		      			result=os.packetout(op);
		      			if(result==0) break;
		      			if(result==-1)
		      			{
		        			if (DEBUG) System.err.println("Corrupt secondary header.  Exiting.");
	                		//return;
	                		//break;// loop;
		      			}
		      			vi.synthesis_headerin(vc, op);
		      			i++;
		    		}
		  		}
		  	}
	        index=oy.buffer(BUFSIZE);
	        buffer=oy.data;
	        try
	        {
				bytes=bitStream.read(buffer, index, BUFSIZE);
			} catch(Exception e)
			  {
	              if (DEBUG) System.err.println(e.getMessage());
			  }
	        if(bytes==0 && i<2)
	        {
		  		if (DEBUG) System.err.println("End of file before finding all Vorbis headers!");
			}
			oy.wrote(bytes);
	      }

	      {
	        byte[][] ptr=vc.user_comments;
	        for(int j=0; j<ptr.length;j++)
	        {
	          if(ptr[j]==null) break;
	          if (DEBUG) System.err.println("Comment: "+new String(ptr[j], 0, ptr[j].length-1));
	        }
	        if (DEBUG) System.err.println("Bitstream is "+vi.channels+" channel, "+vi.rate+"Hz");
	        if (DEBUG) System.err.println("Encoded by: "+new String(vc.vendor, 0, vc.vendor.length-1)+"\n");
	      }

		}
		public void execute()
		{
			if (eos != 0)
			{
				m_circularBuffer.close();
				return;
			}
			try
			{

      //while(eos==0)
      //{
        if ((loop == 1)  && (DEBUG)) System.out.print("|1|");
        //while(eos==0)
        //{
          if (DEBUG) System.out.print("|2|");

	  	  int result=oy.pageout(og);
	  	  if(result==0)
	  	  {
			  if (DEBUG) System.out.print(" More Data 2");
			  readNext();
			  result=oy.pageout(og);
			  loop = 1;
			  //break; // need more data
	      }
	  	  if(result==-1)
	  	  {

			  if (DEBUG) System.out.print(" Currupt 2");
			  // missing or corrupt data at this page position
			  // System.err.println("Corrupt or missing data in bitstream; continuing...");
	      }
	  	  else
	  	  {
            os.pagein(og);
	        // Decoding !
	        while(true)
	        {
	      		result=os.packetout(op);
	      		if(result==0)
	      		{
					if (DEBUG) System.out.print(" More data 3");
					break; // need more data
				}
	      		if(result==-1)
	      		{
					if (DEBUG) System.out.print(" Currupt 3");
					// missing or corrupt data at this page position
		            // no reason to complain; already complained above
	      		}
                else
                {
                	// we have a packet.  Decode it
	        		int samples;
	        		if(vb.synthesis(op)==0)
	        		{
						// test for success!
		  				vd.synthesis_blockin(vb);
					}
	        		while((samples=vd.synthesis_pcmout(_pcmf, _index))>0)
	        		{
  	          			double[][] pcm=_pcm[0];
  	          			float[][] pcmf=_pcmf[0];
                  		boolean clipflag=false;
		  				int bout=(samples<convsize?samples:convsize);

		  				// convert doubles to 16 bit signed ints (host order) and
		  				// interleave
		  				for(i=0;i<vi.channels;i++)
		  				{
		    				int ptr=i*2;
		    				//int ptr=i;
		    				int mono=_index[i];
		    				for(int j=0;j<bout;j++)
		    				{
		      					int val=(int)(pcmf[i][mono+j]*32767.);
		      					if(val>32767)
		      					{
		        					val=32767;
		        					clipflag=true;
		      					}
		      					if(val<-32768)
		      					{
		        					val=-32768;
		        					clipflag=true;
		      					}
                      			if(val<0) val=val|0x8000;
		      					convbuffer[ptr]=(byte)(val);
		      					convbuffer[ptr+1]=(byte)(val>>>8);
		      					ptr+=2*(vi.channels);
		    				}
		  				}
                  		m_circularBuffer.write(convbuffer, 0, 2*vi.channels*bout);
                  		//outputLine.write(convbuffer, 0, 2*vi.channels*bout);
		  				vd.synthesis_read(bout);
	        		}
	      		}
	    	}
	    	loop = 2;
	    	if(og.eos()!=0) eos=1;
	  	  }
        //} // End 2nd while
		if (loop == 1)
		{
        	int n = readNext();
        	if (n == -1) return;
		}
      //}// End 1st while

	    	} catch (Exception e)
	    	  {
				  System.err.println(e.getMessage());
			  }
		}


	/**
	 * Reads next data from input stream.
	 */
	private int readNext()
	{
        	if(eos==0)
        	{
	  			index=oy.buffer(BUFSIZE);
	  			buffer=oy.data;
	  			try
	  			{
					bytes=bitStream.read(buffer,index,BUFSIZE);
					if (DEBUG) System.out.print(" Read3 : "+bytes);
				} catch(Exception e)
				  {
	    			System.err.println(e);
        	    	return -1;
	  			  }
        	  	//if(bytes==-1) break;
	  			oy.wrote(bytes);
	  			if(bytes==0) eos=1;
	  			return bytes;
        	}
        	else return -1;
	}

		/**
		 *
		 */
		private boolean isBigEndian()
		{
			return getFormat().isBigEndian();
		}

		/**
		 *
		 */
		public void close() throws IOException
		{
			super.close();
			bitStream.close();
		}

	}
}

