<?php
///////////////////////////////////////////////////////////////////////////////
//
// IRC client module for PHP IRC clients
//
// This module will handle all major interfacing with the IRC server, making
// all IRC commands easily available by a predefined API.
//
// See phpIRC.inc.php3 for configuration options.
//
// This software and all associated files are released
// under the GNU Public License (GPL), see gpl.txt for details.
//
// Copyright (c) 1999, 2000 by Till Gerken, till@phpwebdev.com
//
///////////////////////////////////////////////////////////////////////////////

define("IRC_MAX_CHANNELS", 8);
define("IRC_MAX_CALLBACKS", 32);
define("IRC_MAX_TIMERS", 2);
class irc_callback
{
	var $code;
	var $function;
}
class irc_timer
{
    var $last_call;
    var $interval;
    var $function;
}
class irc_channel
{
    var $name;
    var $topic;
    var $mode;
    var $nick_list;
}
define("IRCEREG_HOSTNAME", "([a-zA-Z0-9\.-]+)");
define("IRCEREG_SERVER_MESSAGE_NR", "([0-9]{3})");
define("IRCEREG_NICK", "([a-zA-Z0-9\\\^\{\}\[\]\|_`-]+)");
define("IRCEREG_CHANNEL", "([a-zA-Z0-9#_\\\^\{\}\|_-]+)");
define("IRCEREG_COMMAND", "([a-zA-Z]+)");
define("IRCEREG_SPLITUSERHOST", "(.+)!(.+)@(.+)");
define("IRCEREG_WHOISUSER", "(.+) (.+) (.+) \* :(.*)");
define("IRC_NOBREAK", 1000);
define("IRCCB_ONIDLE", 1001);
define("IRCCB_ONKILL", 1002);
define("IRCCB_ONNOTICE", 1100);
define("IRCCB_ONPRIVMSG", 1101);
define("IRCCB_ONJOIN", 1102);
define("IRCCB_ONPART", 1103);
define("IRCCB_ONQUIT", 1104);
define("IRCCB_ONKICK", 1105);
define("IRCCB_ONCHANNELMSG", 1106);
define("IRCCB_ONOP", 1200);
define("IRCCB_ONDEOP", 1201);
define("IRCCB_ONVOICE", 1202);
define("IRCCB_ONDEVOICE", 1203);
define("IRCCB_ONBAN", 1204);
define("IRCCB_ONUNBAN", 1205);
define("IRCCB_ONMODECHANGE", 1300);
define("IRCCB_ONNICKCHANGE", 1400);
define("IRCCB_ONTOPICCHANGE", 1500);
define("IRCCB_ONCTCPPING", 1600);
define("IRCCB_ONACTION", 1700);
define("IRCERR_NOSUCHNICK", 401);
define("IRCERR_NICKNAMEINUSE", 433);
define("IRCERR_INVITEONLYCHAN", 473);
define("IRCRPL_ISON", 303);
define("IRCRPL_AWAY", 301);
define("IRCRPL_WHOISUSER", 311);
define("IRCRPL_WHOISSERVER", 312);
define("IRCRPL_WHOISIRCOP", 313);
define("IRCRPL_WHOISIDLE", 317);
define("IRCRPL_ENDOFWHOIS", 318);
define("IRCRPL_WHOISCHANNELS", 319);
define("IRCRPL_TOPIC", 332);
define("IRCRPL_NAMREPLY", 353);
define("IRCRPL_ENDOFNAMES", 366);
define("IRCRPL_ENDOFMOTD", 376);
$irc_default_nick = "webscript";
$irc_default_realname = "ChildLikeFaith Web Script";
$irc_default_identd = "web";
$irc_valid_identd = 0;
$irc_socket_timeout = 20;
$irc_layer_version = "ChildLikeFaith's Web Script";
$wversion = "1.0.1002";
$ffooter = "ChildLikeFaith's Web Script";

// declare global variables
$irc_connected = 0;				// connected or disconnected?
$irc_server = "";				// server being accessed
$irc_port = 0;					// port being accessed
$irc_nick = "";					// our current nick
$irc_realname = "";				// our current realname
$irc_identd = "";				// our current identd
$irc_userhost = "";				// our current userhost
$irc_usermode = "";				// our current mode
$irc_ircop = 0;					// 1 - we have IRCOp status

$irc_motd = array();				// MOTD of server

$irc_connection_handle = "";			// connection handle with the IRC server

$irc_channel_array = array();			// clear channel array

// statistical variables
$irc_nr_own_messages = 0;			// messages from the network by ourselves
$irc_nr_server_messages = 0;			// messages from the server for us
$irc_nr_regular_messages = 0;			// regular messages

// Callback array for all messages, even channel messages
$irc_callback_array = array();

// Timer array for timed events
$irc_timer_array = array();

// various reply messages exchanged via global variables
$irc_ison_reply = "";				// this variable keeps ISON replies

// WHOIS status variables
$irc_whois_success = 0;				// was the WHOIS successful?
$irc_whois_nick = "";				// nick of the whois target
$irc_whois_identd = "";				// identd of the whois target
$irc_whois_host = "";				// host of the whois target
$irc_whois_realname = "";			// realname of the whois target
$irc_whois_channels = "";			// channels the whois target is participating in
$irc_whois_server = "";				// server the whois target is on
$irc_whois_serverinfo = "";			// serverinfo of that server
$irc_whois_ircop = 0;				// is an ircop?
$irc_whois_idle = 0;				// idle time of whois target
$irc_whois_away = "";				// away reason of whois target

// indicates wether to print out debug messages or not
$irc_debug_flag = 0;

///////////////////////////////////////////////////////////////////////////////
//
// function irc_debug($msg)
//
///////////////////////////////////////////////////////////////////////////////
//
// Prints a debug message
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$msg - message to print
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_nr_own_messages, $irc_nr_server_messages, $irc_nr_regular_messages
//	$irc_debug_flag
//
///////////////////////////////////////////////////////////////////////////////

function irc_debug($msg)
{
	global $irc_nr_own_messages, $irc_nr_server_messages, $irc_nr_regular_messages;
	global $irc_debug_flag;

	if($irc_debug_flag)
	{
		$timestamp = date("H:i:s");
		print("$timestamp - O: $irc_nr_own_messages S: $irc_nr_server_messages R: $irc_nr_regular_messages ||| $msg<br>\n");

		flush();
	}

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_in_array($needle, $haystack)
//
///////////////////////////////////////////////////////////////////////////////
//
// Helper function: searches for a value in an array. Only works with strings.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$needle - value to search for
//	$haystack - array to search in
//
///////////////////////////////////////////////////////////////////////////////
//
// Returns:
//	FALSE - value not found
//	otherwise, key is returned
//
///////////////////////////////////////////////////////////////////////////////

function irc_in_array($needle, $haystack)
{
	
	while(list($key, $value) = each($haystack))
	{
		if(!strcasecmp($value, $needle))
			return(TRUE);
	}
	
	return(FALSE);
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_get_raw_nick()
//
///////////////////////////////////////////////////////////////////////////////
//
// This function returns the "raw" nick, stripping all special characters
// like "@" or "+" from it.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$nick - nick name to treat
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	raw nick
//
///////////////////////////////////////////////////////////////////////////////

function irc_get_raw_nick($nick)
{

	if(($nick[0] == "@") || ($nick[0] == "+"))
		$raw_nick = substr($nick, 1);
	else
		$raw_nick = $nick;

	return($raw_nick);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_is_op($channel, $nick)
//
///////////////////////////////////////////////////////////////////////////////
//
// Determines wether the given nick has operator status in the specififed
// channel.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$channel - channel to look in (must be handled by phpIRC)
//	$nick - nick to test
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	TRUE - nick is an operator
//	FALSE - nick is not an operator
//
///////////////////////////////////////////////////////////////////////////////

function irc_is_op($channel, $nick)
{
	global $irc_channel_array;

	// get channel descriptor index
	$idx = irc_get_channel_by_name($channel);
	if(!$idx)
		return(FALSE);

	// retrieve channel object
	$channel = $irc_channel_array[$idx - 1];
	// get nick list from channel
	$nick_list = $channel->nick_list;

	// make sure we get a plain nickname and prepend
	// the operator character to it
	$nick = "@".irc_get_raw_nick($nick);

	// now check if this element is in the nicklist
	if(!irc_in_array($nick, $nick_list))
		// element is not in nicklist, $nick is not an op
		return(FALSE);
	else
		// element is in nicklist, $nick is an op
		return(TRUE);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_get_message()
//
///////////////////////////////////////////////////////////////////////////////
//
// This function reads the next message from the server. If there's no message
// currently available, it will wait until the server sends something.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	none
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	string containing the server message, 0 on error
//
///////////////////////////////////////////////////////////////////////////////
//
// Globals referred:
//	$irc_connected, $irc_connection_handle, $irc_timer_array
//
///////////////////////////////////////////////////////////////////////////////

function irc_get_message()
{
	global $irc_connected, $irc_connection_handle, $irc_timer_array;
	
	// if we are not connected, quit right away
	if(!$irc_connected)
		return(0);

	// setup our buffer
	$input_buffer = "";

	$done = 0;

	// stuff next character into buffer until we got the current message
	do
	{
		$input_buffer = fgets($irc_connection_handle, 1024);

		if(!$input_buffer)
			irc_do_callback(IRCCB_ONIDLE, "", "", "", "", "");
		else
			$done = 1;

		// get current time
		$current_time = time();
		
		// evaluate timers
		for($i = 0; $i < IRC_MAX_TIMERS; $i++)
		{
			// retrieve timer
			$timer = $irc_timer_array[$i];
			
			// check if timer is in use
			if($timer->interval != -1)
			{
				// timer is in use, do we have to call it?
				if(($current_time - $timer->last_call) >= $timer->interval)
				{
					// have to call timer
					$timer->last_call = $current_time;
					$function = $timer->function;
					$function();
				}
			}
								
			// assign timer back to array
			$irc_timer_array[$i] = $timer;
		}			
	
	} while(!$done && $irc_connected);
	
	return($input_buffer);
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_put_message($message)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function writes a message to the server.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$message - Message to send. It is correctly prefixed and terminated
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	0 - error
//	1 - ok
//
///////////////////////////////////////////////////////////////////////////////
//
// Globals referred:
//	$irc_connected, $irc_connection_handle, $irc_nick
//
///////////////////////////////////////////////////////////////////////////////

function irc_put_message($message)
{
	global $irc_connected, $irc_connection_handle, $irc_nick;
	
	// bail out if we are not connected
	if(!$irc_connected)
		return(0);

	irc_debug("irc_put_message(): Sending \"$message\"");
		
//	fputs($irc_connection_handle, ":$irc_nick $message\n\r");
	fputs($irc_connection_handle, "$message\n\r");
	
	return(1);
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_create_userhost()
//
///////////////////////////////////////////////////////////////////////////////
//
// Creates a userhost string for us and assigns it to the global variable
// $irc_userhost.
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_userhost, $irc_nick, $irc_identd, $irc_valid_identd
//	$irc_host
//
///////////////////////////////////////////////////////////////////////////////

function irc_create_userhost()
{
	global $irc_userhost, $irc_nick, $irc_identd, $irc_valid_identd;
	global $irc_host;
	
	$irc_userhost = $irc_nick . "!";
	
	if(!$irc_valid_identd)
		// if the identd we're supplying is not valid, append a
		// "~"
		$irc_userhost .= "~";
		
	$irc_userhost .= $irc_identd . "@" . $irc_host;
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_get_channel_by_name($name)
//
///////////////////////////////////////////////////////////////////////////////
//
// Searches a channel by its name in the internal channel table and returns
// its handle.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$name - name of channel to search for
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	Channel handle (numeric), 0 on error
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_channel_array
//
///////////////////////////////////////////////////////////////////////////////

function irc_get_channel_by_name($name)
{
	global $irc_channel_array;
	
	for($i = 0; $i < IRC_MAX_CHANNELS; $i++)
	{
		$channel = $irc_channel_array[$i];

		if($channel->name == $name)
			// we found the channel, return it
			return($i + 1);
			
	}
	
	// no channel found, return error
	return(0);
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_set_debug_mode($state)
//
///////////////////////////////////////////////////////////////////////////////
//
// Turns output of debug messages on or off
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$state - 0: do not print debug messages, 1: print debug messages
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	old debug mode
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_debug_flag
//
///////////////////////////////////////////////////////////////////////////////

function irc_set_debug_mode($state)
{
	global $irc_debug_flag;
	
	$old_mode = $irc_debug_flag;
	$irc_debug_flag = $state;
	
	return($old_mode);
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_add_callback($code, $callback)
//
///////////////////////////////////////////////////////////////////////////////
//
// Adds a callback address
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$code - code to call function upon
//	$callback - function to set for callback
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	0 - error
//	1 - ok
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_callback_array
//
///////////////////////////////////////////////////////////////////////////////

function irc_add_callback($code, $callback)
{
	global $irc_callback_array;
	
	// find empty slot for callback
	for($i = 0; $i < IRC_MAX_CALLBACKS; $i++)
	{
		$slot = $irc_callback_array[$i];
		
		if($slot->code == -1)
		{
			// found free slot, initialize it
			$slot->code = $code;
			$slot->function = $callback;
			
			// re-assign slot
			$irc_callback_array[$i] = $slot;
			
			// return success
			return(1);
		}
	}
	
	// no free slot found, return error
	return(0);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_remove_callback($code, $callback)
//
///////////////////////////////////////////////////////////////////////////////
//
// Removes a callback address
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$code - code to call function upon
//	$callback - function to set for callback
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	0 - error
//	1 - ok
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_callback_array
//
///////////////////////////////////////////////////////////////////////////////

function irc_remove_callback($code, $callback)
{
	global $irc_callback_array;
	
	// find slot for callback
	for($i = 0; $i < IRC_MAX_CALLBACKS; $i++)
	{
		$slot = $irc_callback_array[$i];
		
		if(($slot->code == $code) && ($slot->callback == $callback))
		{
			// found slot, kill it
			$slot->code = -1;
			
			// re-assign slot
			$irc_callback_array[$i] = $slot;
			
			// return success
			return(1);
		}
	}
	
	// slot not found, return error
	return(0);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_add_timer($interval, $callback)
//
///////////////////////////////////////////////////////////////////////////////
//
// Adds a timer event
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$interval - interval to call the function at
//	$callback - function to set for callback
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	0 - error
//	1 - ok
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_timer_array
//
///////////////////////////////////////////////////////////////////////////////

function irc_add_timer($interval, $callback)
{
	global $irc_timer_array;
	
	// find empty slot for callback
	for($i = 0; $i < IRC_MAX_TIMERS; $i++)
	{
		$slot = $irc_timer_array[$i];
		
		if($slot->interval == -1)
		{
			// found free slot, initialize it
			$slot->interval = $interval;
			$slot->last_call = time();
			$slot->function = $callback;
			
			// re-assign slot
			$irc_timer_array[$i] = $slot;
			
			// return success
			return(1);
		}
	}
	
	// no free slot found, return error
	return(0);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_remove_timer($interval, $callback)
//
///////////////////////////////////////////////////////////////////////////////
//
// Removes a timer address
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$interval - interval to call function at
//	$callback - function to set for callback
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	0 - error
//	1 - ok
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_timer_array
//
///////////////////////////////////////////////////////////////////////////////

function irc_remove_timer($interval, $callback)
{
	global $irc_timer_array;
	
	// find slot for callback
	for($i = 0; $i < IRC_MAX_TIMERS; $i++)
	{
		$slot = $irc_timer_array[$i];
		
		if(($slot->interval == $interval) && ($slot->callback == $callback))
		{
			// found slot, kill it
			$slot->interval = -1;
			
			// re-assign slot
			$irc_timer_array[$i] = $slot;
			
			// return success
			return(1);
		}
	}
	
	// slot not found, return error
	return(0);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_init()
//
///////////////////////////////////////////////////////////////////////////////
//
// Initializes the IRC library.
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	0 - error
//	1 - ok
//
///////////////////////////////////////////////////////////////////////////////
//
// Globals referred:
//	$irc_connected, $irc_server, $irc_port, $irc_nick, $irc_default_nick
//	$irc_realname, $irc_default_realname, $irc_identd, $irc_default_identd
//	$irc_channel_array, $irc_callback_array, $irc_timer_array
//	$irc_nr_own_messages, $irc_nr_server_messages, $irc_nr_regular_messages
//
///////////////////////////////////////////////////////////////////////////////

function irc_init()
{
	global $irc_connected, $irc_server, $irc_port;
	global $irc_nick, $irc_default_nick, $irc_ircop;
	global $irc_realname, $irc_default_realname;
	global $irc_identd, $irc_default_identd;
	global $irc_channel_array, $irc_callback_array, $irc_timer_array;
	global $irc_nr_own_messages, $irc_nr_server_messages, $irc_nr_regular_messages;
	
	irc_set_debug_mode(0);			// turn off debug messages
	
	$irc_connected = 0;			// we are not connected
	$irc_server = "";			// no server yet
	$irc_port = 0;				// no port yet
	$irc_nick = $irc_default_nick;		// set default nick
	$irc_realname = $irc_default_realname;	// set default realname
	$irc_identd = $irc_default_identd;	// set default identd
	$irc_ircop = 0;				// reset IRCOp status

	$irc_nr_own_messages = 0;
	$irc_nr_server_messages = 0;
	$irc_nr_regular_messages = 0;

	// Initialize callback array
	for($i = 0; $i < IRC_MAX_CALLBACKS; $i++)
	{
		$callback = new irc_callback;
		$callback->code = -1;
		$irc_callback_array[$i] = $callback;
	}

	// initialize timer array
	for($i = 0; $i < IRC_MAX_TIMERS; $i++)
	{
		$timer = new irc_timer;
		$timer->interval = -1;
		$irc_timer_array[$i] = $timer;
	}

	// create a userhost for us temporarily
	irc_create_userhost();

	// initialize channel array
	for($i = 0; $i < IRC_MAX_CHANNELS; $i++)
	{
		$channel = new irc_channel;	// create new channel handle
		
		$channel->name = "";		// mark channel as being unused
		
		$irc_channel_array[$i] = $channel;
	}

	return(1);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_connect_motd_catcher($code, $nick, $identd, $host, $destination,
//				     $text)
//
///////////////////////////////////////////////////////////////////////////////
//
// Callback working on behalf of irc_connect(), stuffing the MOTD into an
// array.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	Usual callback params
//
///////////////////////////////////////////////////////////////////////////////

function irc_connect_motd_catcher($code, $nick, $host, $identd, $destination, $text)
{
	global $irc_motd;

	$irc_motd[] = $text;

}


///////////////////////////////////////////////////////////////////////////////
//
// function irc_connect($server_name, $port_nr, $socket_blocking, $password)
//
///////////////////////////////////////////////////////////////////////////////
//
// Connects to an IRC server and sets up.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$server_name - server to connect to
//	$port_nr - port to connect to
//	$socket_blocking - wether to use blocking sockets (1) or not (0)
//			   defaults to 1
//	$password - password to use (optional)
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	0 - error
//	1 - ok
//
///////////////////////////////////////////////////////////////////////////////
//
// Globals referred:
//	$irc_connected, $irc_server, $irc_port, $irc_connection_handle
//	$irc_nick, $irc_identd, $irc_realname, $irc_host, $irc_motd
//	$irc_socket_timeout, $irc_ircop
//
///////////////////////////////////////////////////////////////////////////////

function irc_connect($server_name, $port_nr, $socket_blocking = 1, $password = "")
{
	global $irc_connected, $irc_server, $irc_port, $irc_connection_handle;
	global $irc_nick, $irc_identd, $irc_realname, $irc_host, $irc_motd;
	global $irc_socket_timeout, $irc_ircop;
	
	// check to see if we are connected, return if so
	if($irc_connected)
		return(0);
		
	// open a socket
	if(!$irc_socket_timeout)
		// no timeout, use compatible sockets
		$irc_connection_handle = fsockopen($server_name, $port_nr);
	else
		$irc_connection_handle = fsockopen($server_name, $port_nr, &$errno, &$errstr, $irc_socket_timeout);
	
	// turn off socket blocking if wanted
	if(!$socket_blocking)
		set_socket_blocking($irc_connection_handle, FALSE);
	
	// return error if connection failed
	if(!$irc_connection_handle)
		return(0);
		
	// ok, we have a connection - now set up

	// send PASS command if required, no backtalk
	if($password != "")
		fputs($irc_connection_handle, "PASS $password\n\r");

	// send USER command, we don't expect backtalk on that one
	fputs($irc_connection_handle, "USER $irc_identd $irc_host $server_name :$irc_realname\n\r");
	
	// now send NICK command, we might get backtalk on this one
	fputs($irc_connection_handle, "NICK $irc_nick\n\r");

	// fake connection status so that irc_get_message() works
	$irc_connected = 1;

	// reset IRCOp status
	$irc_ircop = 0;

	// reset the MOTD-array
	$irc_motd = array();

	// now see if we got an erroneous nick message,
	// if yes, our nick is not correct
	$done = 0;
	while(!$done)
	{
		$msg = irc_get_message();
		// the message format here is:
		// ":<servername> <code> <blah>"
		// The servername is either an IP or a hostname, the blah stuff
		// is not of any interest here. The important thing is the code,
		// it is either a NOTICE or a servermessage number. NOTICEs are
		// unimportant at this point (some servers send them to tell you
		// they're verifying your hostname and stuff)
		$match = explode(" ", $msg);

		// using an ereg here was my last solution (thanks to Spooker),
		// is_int() didn't work and intval() didn't work either		
		if(ereg("^[0-9]+$", $match[1]))
			$done = 1;
		else
		{
			// this might have been a PING message, check for it
			if($match[0] == "PING")
			{
				// yes it was a PING, reply with a PONG
				irc_put_message("PONG $match[1]");
			}
			else
			{
				if($match[0] == "ERROR")
				{
					return(0);
				}
			}
			
			// this wasn't an important server message, get the next one
			// by doing another iteration
		}
	}

	// in case the nickname's in use -> quit
	if($match[1] == IRCERR_NICKNAMEINUSE)
	{
		// send quit message
		irc_put_message("QUIT");
		// close socket
		fclose($irc_connection_handle);
		// unset faked connection status
		$irc_connected = 0;
		// return error
		return(0);
	}

	// remember server name
	$irc_server = substr($match[0], 1);
	
	// set port
	$irc_port = $port_nr;
	
	// assume successful connection now
	$irc_connected = 1;

	irc_debug("Set server to $irc_server, port $irc_port");

	// create userhost to identify our own messages	
	irc_create_userhost();

	// add callback to read the MOTD
	irc_add_callback(372, "irc_connect_motd_catcher");

	// skip the MOTD
	irc_idle(IRCRPL_ENDOFMOTD);

	// remove callback
	irc_remove_callback(372, "irc_connect_motd_catcher");

	// return array with motd
	return($irc_motd);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_disconnect($quit_message)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function just disconnects from IRC.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$quit_message - quit message to send
//
// Return values:
//	0 - error
//	1 - ok
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_connection_handle, $irc_connected
//
///////////////////////////////////////////////////////////////////////////////

function irc_disconnect($quit_message)
{
	global $irc_connection_handle, $irc_connected;
	
	// check if we are connected, otherwise quit
	if(!$irc_connected)
		return(0);
		
	// send quit message
	irc_put_message("QUIT :$quit_message");

	// this is dirty but has to be done to allow
	// the server to settle down
	sleep(5);
	
	// close socket
	fclose($irc_connection_handle);
	
	// reset connection flag
	$irc_connected = 0;
	
	return(1);
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_change_nick($new_nick)
//
///////////////////////////////////////////////////////////////////////////////
//
// Changes the current nickname
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$new_nick - new nick name
//
// Return values:
//	0 - error
//	1 - ok
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_connection_handle, $irc_connected, $irc_nick
//
///////////////////////////////////////////////////////////////////////////////

function irc_change_nick($new_nick)
{
	global $irc_connection_handle, $irc_connected, $irc_nick;
	
	if($irc_connected)
	{
		// we are connected, we have to tell the server that we
		// want to have a new nick
		irc_put_message("NICK $new_nick");
		
		// ************ HACK - check if new nick is accepted! *********
	}
	
	// change nick globally
	$irc_nick = $new_nick;

	// change userhost accordingly
	irc_create_userhost();
	
	return(1);
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_change_realname($new_name)
//
///////////////////////////////////////////////////////////////////////////////
//
// Changes the current realname
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$new_name - new realname
//
// Return values:
//	0 - error
//	1 - ok
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_connected, $irc_realname
//
///////////////////////////////////////////////////////////////////////////////

function irc_change_realname($new_name)
{
	global $irc_connected, $irc_realname;

	// cannot change realname when already connected
	if($irc_connected)
		return(0);

	// were not connected, change realname field
	$irc_realname = $new_name;

	// return success
	return(1);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_oper($nick, $passwd)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function demands oper access
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$nick - nickname in O-Line (doesnt have to be ones own nick)
//	$passwd - password in O-Line
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	none yet (success must be determined later)
//
///////////////////////////////////////////////////////////////////////////////

function irc_oper($nick, $passwd)
{
	global $irc_nick, $irc_ircop;

	// send OPER command
	irc_put_message("OPER $nick $passwd");

	// now retrieve our own whois info, this should determine
	// wether "opering" was successful or not
	$whois_info = irc_whois($irc_nick);

	if($whois_info["ircop"] == 1)
	{
		// were an IRCOp, return success
		$irc_ircop = 1;
		return(1);
	}

	$irc_ircop = 0;

	// nope, were not an IRCOp - return failure
	return(0);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_kill($nick, $reason)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function kills a user from the IRC network (requires OPER access)
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$nick - nick to kill
//	$reason - reason why it was killed
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	0 - error
//	1 - success
//
///////////////////////////////////////////////////////////////////////////////

function irc_kill($nick, $reason)
{
	global $irc_connected, $irc_ircop;

	// check if were connected and have IRCOp status
	if(!$irc_connected || !$irc_ircop)
		// we dont, quit
		return(0);

	// send kill command
	irc_put_message("KILL $nick :$reason");

	return(1);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_join($channel_name, $channel_key)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function joins a channel and returns a handle for it.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$channel_name - name of channel to join
//	$channel_key - key for channel (optional)
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	channel handle, 0 on error
//	
///////////////////////////////////////////////////////////////////////////////
//
// Globals referenced:
//	$irc_connected, $irc_channel_array
//
///////////////////////////////////////////////////////////////////////////////

function irc_join($channel_name, $channel_key = "")
{
	global $irc_connected, $irc_channel_array;

	// return error if we're not connected
	if(!$irc_connected)
		return(0);
		
	// find empty channel slot
	for($i = 0; $i < IRC_MAX_CHANNELS; $i++)
	{
		$channel = $irc_channel_array[$i];
		
		if($channel->name == "")
			break;
	}
		
	// if we didn't find an empty slot, return an error
	if($i == IRC_MAX_CHANNELS)
		return(0);
		
	// now that we have our channel handle, set it up
	$channel->name = $channel_name;
	$channel->topic = "";
	$channel->mode = "";
	$channel->nick_list = array();

	// assign channel back to array
	$irc_channel_array[$i] = $channel;
		
	// send JOIN message to server
	if($channel_key != "")
		irc_put_message("JOIN $channel_name $channel_key");
	else
		irc_put_message("JOIN $channel_name");
	
	return($i + 1);
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_part($channel_name)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function parts a channel and destroys all associated data.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameter:
//	$channel_name - name of channel to part
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	1 on success
//	0 on error
//	
///////////////////////////////////////////////////////////////////////////////
//
// Globals referenced:
//	$irc_connected, $irc_channel_array
//
///////////////////////////////////////////////////////////////////////////////

function irc_part($channel_name)
{
	global $irc_connected, $irc_channel_array;

	// return error if we're not connected
	if(!$irc_connected)
		return(0);
		
	// find channel slot
	for($i = 0; $i < IRC_MAX_CHANNELS; $i++)
	{
		$channel = $irc_channel_array[$i];
		
		if($channel->name == $channel_name)
			break;
	}
		
	// if we didn't find the slot, return an error
	if($i == IRC_MAX_CHANNELS)
		return(0);
		
	// now that we have our channel slot, mark it as free
	$channel->name = "";
	$channel->topic = "";
	$channel->mode = "";
	$channel->nick_list = array();

	// assign channel back to array
	$irc_channel_array[$i] = $channel;
		
	// send PART message to server
	irc_put_message("PART $channel_name");
	
	return($i + 1);
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_ison($nick)
//
///////////////////////////////////////////////////////////////////////////////
//
// Sends an ISON query to the server to find out if a nick is currently online.
// Warning: this function idles until a reply is received!
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$nick - nickname to check wether if he/she/it is online or not
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	0 - nick is not online
//	1 - nick is online
//
///////////////////////////////////////////////////////////////////////////////
//
// Globals referenced:
//	$irc_ison_reply
//
///////////////////////////////////////////////////////////////////////////////

function irc_ison($nick)
{
	global $irc_ison_reply;

	// get real nick
	$nick = irc_get_raw_nick($nick);

	// send ISON command	
	irc_put_message("ISON $nick");

	// idle, waiting for reply
	irc_idle(IRCRPL_ISON);

	// in case the reply matches our request, return success
	if(strtoupper($irc_ison_reply) == strtoupper($nick))
		return(1);

	return(0);
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_whois($nick)
//
///////////////////////////////////////////////////////////////////////////////
//
// Sends a WHOIS query to the server to find out details about a nick.
// Warning: this function idles until a reply is received!
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$nick - nickname to retrieve info about
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	0 - nick is not online
//	1 - nick is online
//
///////////////////////////////////////////////////////////////////////////////
//
// Globals referenced:
//	$irc_whois_success, $irc_whois_nick, $irc_whois_identd, $irc_whois_host
//	$irc_whois_realname, $irc_whois_channels, $irc_whois_server
//	$irc_whois_serverinfo, $irc_whois_idle, $irc_whois_away
//
///////////////////////////////////////////////////////////////////////////////

function irc_whois($nick)
{
	global $irc_whois_success, $irc_whois_nick, $irc_whois_identd;
	global $irc_whois_host, $irc_whois_realname, $irc_whois_channels;
	global $irc_whois_server, $irc_whois_serverinfo, $irc_whois_idle;
	global $irc_whois_ircop, $irc_whois_away;

	// get raw nick
	$nick = irc_get_raw_nick($nick);
	
	// assume we were successful
	$irc_whois_success = 1;
	
	// clear all variables
	$irc_whois_nick = "";
	$irc_whois_identd = "";
	$irc_whois_host = "";
	$irc_whois_realname = "";
	$irc_whois_channels = "";
	$irc_whois_server = "";
	$irc_whois_serverinfo = "";
	$irc_whois_ircop = 0;
	$irc_whois_idle = 0;
	$irc_whois_away = "";

	if(!irc_ison($nick))
		return(0);

	// send WHOIS command	
	irc_put_message("WHOIS $nick");

	irc_idle(IRCRPL_ENDOFWHOIS);

	// if whois failed, return error
	if(!$irc_whois_success)
		return(0);
	
	// whois was successful, construct array
	$result = array("nick" => $irc_whois_nick,
			"identd" => $irc_whois_identd,
			"host" => $irc_whois_host,
			"realname" => $irc_whois_realname,
			"channels" => $irc_whois_channels,
			"server" => $irc_whois_server,
			"serverinfo" => $irc_whois_serverinfo,
			"ircop" => $irc_whois_ircop,
			"idle" => $irc_whois_idle,
			"away" => $irc_whois_away);
	
	return($result);
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_privmsg($destination, $msg)
//
///////////////////////////////////////////////////////////////////////////////
//
// Sends out a private message (i.e. PRIVMSG) to a channel or a nick. Sending
// out to a channel will only work by participating in the channel itself or
// if the destination channels allows external messages. Sending out to a nick
// will open a query window on the other end.
//
// You can also specify hostmasks if it is allowed and you have the according
// privileges.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$destination - destination channel, nick or hostmask (list)
//	$msg - message text
//
///////////////////////////////////////////////////////////////////////////////

function irc_privmsg($destination, $msg)
{

	// get raw nick name, if nickname
	$destination = irc_get_raw_nick($destination);

	irc_put_message("PRIVMSG $destination :$msg");

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_notice($destination, $msg)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function sends out a notice. See also irc_privmsg().
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$destination - destination channel, nick, or hostmask (list)
//	$msg - message text
//
///////////////////////////////////////////////////////////////////////////////

function irc_notice($destination, $msg)
{

	// get raw nick, if nick
	$destination = irc_get_raw_nick($destination);
	
	irc_put_message("NOTICE $destination :$msg");

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_action($destination, $message)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function sends out an "action" message to the specified destination.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$destination - destination to send action to
//	$message - message to include
//
///////////////////////////////////////////////////////////////////////////////

function irc_action($destination, $message)
{

	// get raw nick, if nick
	$destination = irc_get_raw_nick($destination);

	irc_privmsg($destination, irc_ctcp_quote("ACTION $message"));

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_set_mode($destination, $mode, $parameter)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function is used to set modes. It can be used to set any mode, however,
// it is not encouraged to be used by the user but only from within phpIRC as
// using the API for setting special modes is considered to be safer.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$destination - destination channel or nick for mode (only valid nick is
//	               our own nick)
//	$mode - mode to set (string)
//	$parameter - parameters for mode, MUST be formatted correctly!
//
///////////////////////////////////////////////////////////////////////////////

function irc_set_mode($destination, $mode, $parameter)
{

	// get raw nick, if nick
	$destination = irc_get_raw_nick($destination);
	$parameter = irc_get_raw_nick($parameter);
	
	irc_put_message("MODE $destination $mode $parameter");
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_kick($destination, $nick, $comment)
//
///////////////////////////////////////////////////////////////////////////////
//
// Kicks $nick from $destination
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$destination - where to kick the nick
//	$nick - nick to kick
//	$comment - comment to add to kick message
//
///////////////////////////////////////////////////////////////////////////////

function irc_kick($destination, $nick, $comment)
{

	// get raw nick
	$destination = irc_get_raw_nick($destination);

	irc_put_message("KICK $destination $nick :$comment");

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_ctcp_ping($nick)
//
///////////////////////////////////////////////////////////////////////////////
//
// CTCP PINGs someone
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$nick - nick to ping
//
///////////////////////////////////////////////////////////////////////////////

function irc_ctcp_ping($nick)
{

	// get raw nick, if nick
	$nick = irc_get_raw_nick($nick);
	
	$message = irc_ctcp_quote("PING ".time());
	
	irc_privmsg($nick, $message);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_get_nick_list($channel_name)
//
///////////////////////////////////////////////////////////////////////////////
//
// Returns the nick list for the given channel.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$channel - channel name (string)
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	array containing all nicks participating in that channel (non-consecutive)
//	0 on error (can only return nick lists of channels we're participating in)
//
///////////////////////////////////////////////////////////////////////////////

function irc_get_nick_list($channel_name)
{
	global $irc_channel_array;
	
	$idx = irc_get_channel_by_name($channel_name);
	if(!$idx)
		// didn't find channel, return error
		return(0);
	
	$channel = $irc_channel_array[$idx - 1];
	
	return($channel->nick_list);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_do_callback($code, $nick, $identd, $host, $destination, $text)
//
///////////////////////////////////////////////////////////////////////////////
//
// Calls all callbacks for the given code.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$code - Callback code
//	$nick - Originating nick
//	$identd - Originating identd
//	$host - Originating host
//	$destination - Destination the message/event was sent to (channel/nick)
//	$text - Message text (can be empty, depending on callback type)
//
///////////////////////////////////////////////////////////////////////////////
//
// Globals referenced:
//	$irc_callback_array
//
///////////////////////////////////////////////////////////////////////////////

function irc_do_callback($code, $nick, $identd, $host, $destination, $text)
{
	global $irc_callback_array;

	// search through whole callback array	
	for($i = 0; $i < IRC_MAX_CALLBACKS; $i++)
	{
		// get this callback's info
		$slot = $irc_callback_array[$i];
		
		// does it match the requested code?
		if($slot->code == $code)
		{
			// codes are matching, retrieve function address
			$callback = $slot->function;
			
			// call it
			$callback($code, $nick, $identd, $host, $destination, $text);
		}
	}
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_handle_server_message($msg, $break_condition)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function parses a server message and acts accordingly.
//
// Do not call this function directly, it should only be called by irc_idle()
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$msg - the message to parse
//	$break_condition - break condition to check against
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	0 - break condition not met
//	1 - break condition met
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_nr_server_messages, $irc_channel_array
//	$irc_ison_reply, $irc_whois_success, $irc_whois_nick, $irc_whois_identd
//	$irc_whois_host, $irc_whois_realname, $irc_whois_channels
//	$irc_whois_server, $irc_whois_serverinfo, $irc_whois_idle
//	$irc_whois_away
//
///////////////////////////////////////////////////////////////////////////////

function irc_handle_server_message($msg, $break_condition)
{
	global $irc_nr_server_messages, $irc_channel_array;
	global $irc_ison_reply;
	global $irc_whois_success, $irc_whois_nick, $irc_whois_identd;
	global $irc_whois_host, $irc_whois_realname, $irc_whois_channels;
	global $irc_whois_server, $irc_whois_serverinfo, $irc_whois_idle;
	global $irc_whois_ircop, $irc_whois_away;
	
	// update statistics
	$irc_nr_server_messages++;

	// the message received here is of the format
	// <msg_nr> <destination> <rest>
	// The message number is a 3-digit number indicating the message type.
	// The destination should always be the client's nick.
	// The rest of the message will usually be of a similar form, but
	// some messages require different handling.

	$match = explode(" ", $msg);
	
	// skip all server notices or different things, just handle numeric msgs
	if(!ereg("^[0-9]+$", $match[0]))
	{
		irc_debug("irc_handle_server_message(): Skipping textual message \"$msg\"");
		return(0);
	}

	// now divide the message in its parts one by one
	
	// get message number
	$msg_nr = $match[0];
	
	// this one's not really needed, it's always our nick
	$destination = $match[1];
	
	$rest = substr($msg, strlen($match[0]) + 1 + strlen($match[1]) + 1);

	// do not modify $rest in this switch()-statement, it is needed later on
	switch($msg_nr)
	{
		case IRCRPL_ISON:	// received an ISON reply
					// extract reply to global variable
					$irc_ison_reply = chop(substr($rest, 1, strlen($rest) - 1));
					
					break;
					
		case IRCRPL_TOPIC:	// received a topic
					// <channel> :<new topic>
					ereg(IRCEREG_CHANNEL." :(.*)", $rest, $match);
					
					$channel_name = $match[1];
					$topic = $match[2];
					
					irc_debug("irc_handle_server_message(): Setting topic for channel $channel_name to \"$topic\"");

					// get channel handle					
					$channel_handle = irc_get_channel_by_name($channel_name);
					
					// set new topic if channel exists
					if($channel_handle)
					{
						// assign topic
						$channel = $irc_channel_array[$channel_handle - 1];
						$channel->topic = $topic;
						$irc_channel_array[$channel_handle - 1] = $channel;
					}
					
					break;

		case IRCRPL_NAMREPLY:	// receiving nicklist for a channel
					// = <channel> :<nick list>
					$match = explode(" ", $rest);
					$channel_name = $match[1];
					
					$nick_list = substr($rest, strrpos($rest, ":") + 1);
					$nick_list = trim($nick_list);
					
					irc_debug("irc_handle_server_message(): Setting nicklist for channel $channel_name to \"$nick_list\"");
					
					// get channel handle
					$channel_handle = irc_get_channel_by_name($channel_name);
					
					// set new nick list if channel exists
					if($channel_handle)
					{
						// split nicklist into an array
						$nick_list = explode(" ", $nick_list);
						
						// get previous nick list (MUST be cleared by irc_join())
						$channel = $irc_channel_array[$channel_handle - 1];
						$channel_list = $channel->nick_list;
						
						for($i = 0; $i < count($nick_list); $i++)
							if(!irc_in_array($nick_list[$i], $channel_list))
								$channel_list[] = $nick_list[$i];
						
						$channel->nick_list = $channel_list;
						
						$irc_channel_array[$channel_handle - 1] = $channel;
					}
					
					break;
					
		case IRCRPL_ENDOFNAMES:	// end of names list, ignore this message
					break;

		case IRCRPL_AWAY:	// away reply for whois query
					// cut out prefix
					$tmp = strchr($rest, " :");
					
					$irc_whois_away = substr($tmp, 2, strlen($tmp) - 2);
					
					break;

		case IRCRPL_WHOISUSER:	// user reply for whois query
					ereg(IRCEREG_WHOISUSER, $rest, $match);

					// set global variables
					$irc_whois_nick = $match[1];
					$irc_whois_identd = str_replace("~", "", $match[2]);
					$irc_whois_host = $match[3];
					$irc_whois_realname = $match[4];
					
					break;

		case IRCRPL_WHOISCHANNELS:
					// channel reply for whois query
					// ****** HACK: This one relies on $irc_whois_nick already been set! ********
					
					// cut prefix out of string
					$tmp = str_replace($irc_whois_nick." :", " ", $rest);
					
					// cut operator or voice prefixes from channel names
//					$rest = str_replace(" +#", "", $rest);
//					$rest = str_replace(" @#", "", $rest);
					
					// cut whitespace from beginning and end of string and assign channel list
					$irc_whois_channels = trim($tmp);
					
					break;
		
		case IRCRPL_WHOISSERVER:
					// server reply for whois query
					// ****** HACK: This one relies on $irc_whois_nick already been set! ********
					// cut out nick
					$tmp = strchr($rest, " ");
					
					// extract server information
					$irc_whois_serverinfo = strchr($tmp, ":");
					
					// cut out ":"
					$irc_whois_serverinfo = substr($irc_whois_serverinfo, 1, strlen($irc_whois_serverinfo) - 1);
					
					// extract server name from message
					$irc_whois_server = substr($tmp, 0, strlen($tmp) - strlen($irc_whois_serverinfo) - 2);
					
					break;

		case IRCRPL_WHOISIRCOP:	// this reply is only sent if the queried
					// person is an IRC Operator, so we can
					// safely set this flag to 1 here
					$irc_whois_ircop = 1;

					break;

		case IRCRPL_WHOISIDLE:	// idle reply for whois query
					// cut out nick
					$tmp = strchr($rest, " ");
					
					// extract idle time, skip the rest (dirty, but works)
					$irc_whois_idle = (int)$tmp;
					
					break;
		
//		case IRCERR_INVITEONLYCHAN:
//					// channel is invite only, send status report
//					// the server message has the format
//					// <channel> :<errormessage>
//					ereg(IRCEREG_CHANNEL." :(.*)", $rest, $match);
//
//					irc_debug("irc_handle_server_message(): Initiating status callback, channel $match[1] is invite only (code ".IRCERR_INVITEONLYCHAN.")");
//					
//					break;

		case IRCERR_NOSUCHNICK:	// error message, no such nick was found on the network
					// this is especially important for WHOIS queries
					// if this was due to a whois, the whois failed
					$irc_whois_success = 0;
					break;

		default:		// unhandled message
					irc_debug("irc_handle_server_message(): Unhandled message nr. $msg_nr, txt: $rest");
					break;
	}

	// call all relevant callbacks
	// these are raw callbacks, no preprocessing has been done on these messages
	// in case you need preprocessed callbacks, don't use servermessages as events
	irc_do_callback($msg_nr, "", "", "", "", $rest);

	// now handle break condition
	if($break_condition == $msg_nr)
	{
		// we have encountered a break condition, announce it
		irc_debug("irc_handle_server_message(): Break condition $break_condition encountered");
		
		return(1);
	}
	
	return(0);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_handle_mode_message($nick, $identd, $host, $msg)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function handles all mode messages
//
// Do not call this function directly, it should only be called by
// irc_handle_own_message() and irc_handle_regular_message()
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$nick - originating nick
//	$identd - originating identd
//	$host - originating host
//	$msg - mode message
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_nick, $irc_usermode, $irc_channel_array
//
///////////////////////////////////////////////////////////////////////////////

function irc_handle_mode_message($nick, $identd, $host, $msg)
{
	global $irc_nick, $irc_usermode, $irc_channel_array;

	// the format of the message at this stage is
	// <command> <destination>  <mode string> <parameter>
	// command is equal to "MODE" here
	// destination is the nick of the destination
	// mode string is the mode string
	// the parameter list consists of a list of mode parameters
	$match = explode(" ", $msg);
	
	$command = $match[0];
	$destination = $match[1];

	// get mode string
	$mode_str = trim($match[2]);
				
	// determine which mode to modify - user or channel mode?
	if(!strcasecmp($destination, $irc_nick))
	{
		// this one is a user mode
		
		// we have to scan the mode string to analyze the new mode
		$mode = 1;	// 1 = set, 0 = clear
		
		for($i = 1; $i < strlen($mode_str); $i++)
		{
			switch($mode_str[$i])
			{
				case "+":	// set mode to "set"
						$mode = 1;
						break;
								
				case "-":	// set mode to "clear"
						$mode = 0;
						break;
								
				default:	// must be a mode flag, set/clear
						if(!$mode)
							// remove character
							$irc_usermode = str_replace($mode_str[$i], "", $irc_usermode);
						else
							// add character
							$irc_usermode .= $mode_str[$i];
							
						break;
			}
		}
					
		irc_debug("irc_handle_mode_message(): New usermode is now \"$irc_usermode\"");
					
	}
	else
	{
		// it's a channel mode
					
		// get index of first parameter
		$parameter_idx = 3;

		// get channel index
		$idx = irc_get_channel_by_name($destination);
					
		if(!$idx)
//			break;
			return;
					
		$channel = $irc_channel_array[$idx - 1];
		$channel_mode = $channel->mode;
					
		$mode = 1;	// 1 = set, 0 = clear
					
		// now scan mode string
		for($i = 0; $i < strlen($mode_str); $i++)
		{
			switch($mode_str[$i])
			{
				case "+":	// set mode to "set"
						$mode = 1;
						break;
								
				case "-":	// set mode to "clear"
						$mode = 0;
						break;
							
				case "n":
				case "m":
				case "t":
				case "i":
				case "n":	// must be a mode flag, set/clear
						if(!$mode)
							// remove character
							$channel_mode = str_replace($mode_str[$i], "", $channel_mode);
						else
							// add character
							$channel_mode .= $mode_str[$i];

						$channel->mode = $channel_mode;
						$irc_channel_array[$idx - 1] = $channel;
						
						irc_debug("irc_handle_mode_message(): $nick!$identd@$host sets new channel mode for $destination to \"$channel_mode\"");
						irc_do_callback(IRCCB_ONMODECHANGE, $nick, $identd, $host, $destination, $channel_mode);
						break;

				case "b":	// sets or removes a ban, needs a parameter
						$param = $match[$parameter_idx];
						$parameter_idx++;
						
						if($mode)
						{
							irc_debug("irc_handle_mode_message(): $nick!$identd@$host has banned $param from $destination");
							irc_do_callback(IRCCB_ONBAN, $nick, $identd, $host, $destination, $param);
						}
						else
						{
							irc_debug("irc_handle_mode_message(): $nick!$identd@$host has unbanned $param from $destination");
							irc_do_callback(IRCCB_ONUNBAN, $nick, $identd, $host, $destination, $param);
						}
						
						break;
						
				case "o":	// this one sets ops, needs a parameter
						$nick_list = $channel->nick_list;
						
						while(list($key, $target) = each($nick_list))
						{
                            $target = irc_get_raw_nick($target);
									
							if($target == $match[$parameter_idx])
							{
								// found target, decide clear/set
								if($mode)
								{
									irc_debug("irc_handle_mode_message(): $nick!$identd@$host sets operator status for $target on $destination");
											
									$nick_list[$key] = "@".$target;
									$channel->nick_list = $nick_list;
									$irc_channel_array[$i] = $channel;
									
									$channel->mode = $channel_mode;
									$irc_channel_array[$idx - 1] = $channel;

									// call callbacks
									irc_do_callback(IRCCB_ONOP, $nick, $identd, $host, $destination, $target);
								}
								else
								{
									irc_debug("irc_handle_mode_message(): $nick!$identd@$host clears operator status for $target on $destination");
									
									$nick_list[$key] = $target;
									$channel->nick_list = $nick_list;
									$irc_channel_array[$i] = $channel;
									
									$channel->mode = $channel_mode;
									$irc_channel_array[$idx - 1] = $channel;


									// call callbacks
									irc_do_callback(IRCCB_ONDEOP, $nick, $identd, $host, $destination, $target);
								}
							}
						}
									
						$parameter_idx++;
									
						break;
									
				case "v":	// this one sets voice, needs a parameter
						$nick_list = $channel->nick_list;

						// only go through this if target is not an operator already
						if(!irc_is_op($destination, irc_get_raw_nick($match[$parameter_idx])))
						{

							while(list($key, $target) = each($nick_list))						
							{
                                $target = irc_get_raw_nick($target);
									
								if($target == $match[$parameter_idx])
								{
									// found target, decide clear/set
									if($mode)
									{
										irc_debug("irc_handle_mode_message(): $nick!$identd@$host sets voice status for $target on $destination");
											
										$nick_list[$key] = "+".$target;
										$channel->nick_list = $nick_list;
										$irc_channel_array[$i] = $channel;
									
										$channel->mode = $channel_mode;
										$irc_channel_array[$idx - 1] = $channel;

										// call callbacks
										irc_do_callback(IRCCB_ONVOICE, $nick, $identd, $host, $destination, $target);
									}
									else
									{
										irc_debug("irc_handle_mode_message(): $nick!$identd@$host clears voice status for $target on $destination");
									
										$nick_list[$key] = $target;
										$channel->nick_list = $nick_list;
										$irc_channel_array[$i] = $channel;
									
										$channel->mode = $channel_mode;
										$irc_channel_array[$idx - 1] = $channel;

										// call callbacks
										irc_do_callback(IRCCB_ONDEVOICE, $nick, $identd, $host, $destination, $target);
									}
								}
							}
						}

						$parameter_idx++;
									
						break;
						
				default:	// unknown mode
						irc_debug("irc_handle_mode_message(): Unknown mode $mode_str[$i]");
						break;
			}
		}
		
		$channel->mode = $channel_mode;
		$irc_channel_array[$idx - 1] = $channel;

	}

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_handle_own_message($msg)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function parses a message sent from us and acts accordingly.
//
// Do not call this function directly, it should only be called by irc_idle()
//
// NOTE: This function has been disabled! There is currently absolutely NO
//	 justification for differentiating between own and regular messages.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$msg - message to be parsed
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_nr_own_messages, $irc_nick, $irc_identd, $irc_host
//
///////////////////////////////////////////////////////////////////////////////

function irc_handle_own_message($msg)
{
	global $irc_nr_own_messages, $irc_nick, $irc_identd, $irc_host;
	
	// update statistics
	$irc_nr_own_messages++;

	// the format of the message at this stage is
	// <command> <destination> :<parameter>
	// command is any of the commands you can issue, in upper case
	// destination is the nick of the destination
	// the parameter list is usually a text, consisting of any words
	$command = substr($msg, 0, strpos($msg, " "));

	switch($command)
	{
		case "MODE":	// set a new mode - either channel mode or user mode
				// format of this message:
				// "MODE <destination> <new mode> <parameters>"
				// call a dedicated function for this
				irc_handle_mode_message($irc_nick, $irc_identd, $irc_host, $msg);
				break;
				
		default:	// unknown message
				irc_debug("irc_handle_own_message(): Unhandled message \"$msg\"");
				break;
	}
	
}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_ctcp_quote($message)
//
///////////////////////////////////////////////////////////////////////////////
//
// "Quotes" a CTCP message, i.e. it will encode a message so that it is suitable
// for CTCP sending it to a client.
//
// Warning: this function will not quote very efficient and correctly, it just
// replaces special characters with their quoted equivalents and might (or,
// probably will) produce invalid messages when sending sophisticated data
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$message - CTCP message to quote
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	encoded message (string)
//
///////////////////////////////////////////////////////////////////////////////

function irc_ctcp_quote($message)
{

	// add middle-quoted characters
	$result = str_replace(chr(16), chr(16).chr(16), $message);
	$result = str_replace("\r", chr(16)."r", $result);
	$result = str_replace("\n", chr(16)."n", $result);
	$result = str_replace(chr(0), chr(16)."0", $result);

	$result = str_replace("\\", "\\\\", $result);
	$result = str_replace(chr(1), "\\a", $result);

	// VERY, very basic handling of protocol - just add outer quotes
	$result = chr(1).$result.chr(1);
	
	return($result);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_ctcp_unquote($message)
//
///////////////////////////////////////////////////////////////////////////////
//
// "Unquotes" a CTCP message, i.e. it will restore the original content from a
// message that has been binary encoded for transfer using the CTCP protocol.
//
// Warning: this function cannot handle incorrectly quoted messages and can
// only do VERY basic dequoting, as it only replaces quoted characters with
// their dequoted equivalents instead of following the extraction chain
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$message - quoted CTCP message to decode
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	decoded message (string)
//
///////////////////////////////////////////////////////////////////////////////

function irc_ctcp_unquote($message)
{

	// remove middle-quoted characters
	$result = str_replace(chr(16)."0", chr(0), $message);
	$result = str_replace(chr(16)."n", "\n", $result);
	$result = str_replace(chr(16)."r", "\r", $result);
	$result = str_replace(chr(16).chr(16), chr(16), $result);

	$result = str_replace("\\a", chr(1), $result);
	$result = str_replace("\\\\", "\\", $result);

	// VERY, very basic handling of protocol - just remove outer quotes
	$result = substr($result, 1, strlen($result) - 2);
	
	return($result);

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_handle_ctcp_message($nick, $identd, $host, $destination, $text)
//
///////////////////////////////////////////////////////////////////////////////
//
// Takes care of all CTCP'ed messages
//
// Do not call this function directly, it should only be called by
// irc_handle_regular_message()
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$nick - originating nick
//	$identd - originating identd
//	$host - originating host
//	$destination - destination of message
//	$text - message text
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_layer_version
//
///////////////////////////////////////////////////////////////////////////////

function irc_handle_ctcp_message($nick, $identd, $host, $destination, $text)
{
	global $irc_layer_version;

	// unquote message
	$message = irc_ctcp_unquote($text);

	// get message parts
	$match = explode(" ", $message);

	$command = strtoupper($match[0]);

	switch($command)
	{
		case "CLIENTINFO": // clientinfo request, reply with all commands that are understood
				// this is hardcoded here, bad practice, but too much effort to do it
				// differently (I believe this must be ordered alphabetically)
				irc_notice($nick, irc_ctcp_quote("CLIENTINFO PING TIME VERSION"));
				
				irc_debug("irc_handle_ctcp_message(): Clientinfo request from $nick!$identd@$host");
				
				break;
				
		case "PING":	// somebody is pinging us, we have to send the same message back
				// don't use requoting again as we can safely use the original contents
				// of the message to save processing time
				irc_notice($nick, $text);
				
				irc_debug("irc_handle_ctcp_message(): PING? PONG! (from $nick!$identd@$host)");
				
				irc_do_callback(IRCCB_ONCTCPPING, $nick, $identd, $host, $destination, $text);
				break;

		case "TIME":	// TIME request, answer with current system time and date
				// Reply format: "Tuesday 08/24/1999 17:47:58"
				$reply = date("l m/d/Y H:i:s");
				irc_notice($nick, irc_ctcp_quote("TIME ".$reply));
				
				irc_debug("irc_handle_ctcp_message(): Time request from $nick!$identd@$host, replied \"$reply\"");
				
				break;
				
		case "VERSION":	// client version request
				irc_notice($nick, irc_ctcp_quote("VERSION ".$irc_layer_version));
				
				irc_debug("irc_handle_ctcp_message(): Version request from $nick!$identd@$host");
				
				break;
		
		case "ACTION":	// action message, used in channels
				// extract the action text from the message
				$action_text = trim(strchr($message, " "));
				
				irc_debug("irc_handle_ctcp_message(): Action from $nick!$identd@$host in $destination - \"$action_text\"");
				
				irc_do_callback(IRCCB_ONACTION, $nick, $identd, $host, $destination, $action_text);
				
				break;
		
		default:	// unhandled message
				irc_debug("irc_handle_ctcp_message(): Unhandled CTCP message from $nick!$identd@$host for $destination, \"$message\"");
				
				break;
	}

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_handle_regular_message($origin, $msg)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function parses a regular IRC message (should be about 80% of the traffic).
//
// Do not call this function directly, it should only be called by irc_idle()
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$origin - originating address of message
//	$msg - message to be parsed
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_nr_regular_messages, $irc_nick, $irc_channel_array, $irc_connected
//
///////////////////////////////////////////////////////////////////////////////

function irc_handle_regular_message($origin, $msg)
{
	global $irc_nr_regular_messages, $irc_nick, $irc_channel_array;
	global $irc_connected;
	
	// update statistics
	$irc_nr_regular_messages++;
	
	// this message is of the format
	// "<command> <destination> :<text>"
	// the command would usually be PRIVMSG,
	// the destination either a channel or our own nick
	// difference for KICK:
	// "<command> <destination> <kick_nick> :<reason>"
	$match = explode(" ", $msg);
	$command = strtoupper($match[0]);
	$destination = $match[1];
	
	if(isset($match[2]))
		$kick_nick = $match[2];				// this one only needed for kicks,
								// invalid otherwise!

	$text = strchr($msg, ":");
	$text = trim(substr($text, 1));
	
	// split origin into nick, identd and host
	ereg(IRCEREG_SPLITUSERHOST, $origin, $match);
	
	$nick = $match[1];
	$identd = $match[2];
	$host = $match[3];
	
	// strip the "invalid identd" tag ("~") from identd
	$identd = str_replace("~", "", $identd);

	switch($command)
	{
		case "PRIVMSG":	// handle a private message - this means both channel messages and queries
		
				// first of all, find out if it's a CTCP'ed message
				if($text[0] == chr(1))
				{
					// alright, this is a CTCP - let another procedure handle it
					irc_handle_ctcp_message($nick, $identd, $host, $destination, $text);
				}
				else
				{
					// this is either a query or a channel message, determine
					// callback type
					if(!strcasecmp($destination, $irc_nick))
					{
						// must be a private message
						$callback_code = IRCCB_ONPRIVMSG;
					}
					else
					{
						// this is probably a channel message since
						// we do not receive queries to others as a client
						$callback_code = IRCCB_ONCHANNELMSG;
					}
					
					// call all callbacks
					irc_debug("irc_handle_regular_message(): Passing PRIVMSG from $nick!$identd@$host for \"$destination\" on to callback, Code $callback_code, Text \"$text\"");
					irc_do_callback($callback_code, $nick, $identd, $host, $destination, $text);
				}
				
				break;

		case "NOTICE":	// handle a NOTICE - same as private message, instead of that no
				// automatic replies are allowed for this one
				
				// this is either a private notice or a channel notice,
				// we do not differentiate here though
				
				// first of all, find out if it's a CTCP reply
				if($text[0] == chr(1))
				{
					// alright, this is a CTCP reply
					// we have no handlers for this yet
					irc_debug("irc_handle_regular_message(): Unhandled CTCP reply from $nick!$identd@$host for \"$destination\", Text \"$text\"");
				}
				else
				{
					// regular notice, pass it on to callback
					irc_debug("irc_handle_regular_message(): Passing NOTICE from $nick!$identd@$host for \"$destination\" on to callback, Code ".IRCCB_ONNOTICE.", Text \"$text\"");
					irc_do_callback(IRCCB_ONNOTICE, $nick, $identd, $host, $destination, $text);
				}

				break;

		case "JOIN":	// JOIN message, format:
				// "JOIN :<channel>"
				$destination = $text;

				$idx = irc_get_channel_by_name($destination);
				
				// break on invalid index
				if(!$idx)
					break;
				
				$channel = $irc_channel_array[$idx - 1];
				
				// found channel slot, add nick to list
				$nick_list = $channel->nick_list;
				
				// add nick to list if not already in array
				// ...and if it is not ourselves!
				if(($nick != $irc_nick) && !irc_in_array($nick, $nick_list))
					$nick_list[] = $nick;
				
				$channel->nick_list = $nick_list;
				$irc_channel_array[$idx - 1] = $channel;
				
				// call appropriate callbacks
				irc_debug("irc_handle_regular_message(): JOIN Message passed on to callback - Code ".IRCCB_ONJOIN.", $nick!$identd@$host joined \"$destination\"");
				irc_do_callback(IRCCB_ONJOIN, $nick, $identd, $host, $destination, "");
				
				break;

		case "MODE":	// MODE message, format
				// "MODE <channel> <mode_str> <parameters>
				irc_handle_mode_message($nick, $identd, $host, $msg);
				break;

		case "PART":	// PART message, format:
				// "PART <channel> :<reason>"
				$destination = trim($destination);
				$idx = irc_get_channel_by_name($destination);

				// break on invalid index
				if(!$idx)
					break;
				
				$channel = $irc_channel_array[$idx - 1];
				
				// found channel slot, remove nick from list
				$nick_list = $channel->nick_list;
				
				while(list($key, $target) = each($nick_list))
					if(!strcasecmp(irc_get_raw_nick($target), $nick))
						unset($nick_list[$key]);
				
				irc_debug("Nick list for channel $channel->name AFTER part:");
				while($nickname = each($nick_list))
					irc_debug("\"$nickname[1]\"");
				
				$channel->nick_list = $nick_list;
				$irc_channel_array[$idx - 1] = $channel;

				// call appropriate callbacks
				irc_debug("irc_handle_regular_message(): PART Message passed on to callback - Code ".IRCCB_ONPART.", $nick!$identd@$host left \"$destination\"");
				irc_do_callback(IRCCB_ONPART, $nick, $identd, $host, $destination, "");
				
				break;

		case "QUIT":	// QUIT message, format:
				// "QUIT :<reason>"
				
				// find all channels that contain this nick and
				// delete it from the nick list
				for($i = 0; $i < IRC_MAX_CHANNELS; $i++)
				{
					$channel = $irc_channel_array[$i];
					
					if($channel->name != "")
					{
						// this channel slot is in use, check if nick
						// is in nick list
						$nick_list = $channel->nick_list;
						
						while(list($key, $target) = each($nick_list))
							if(!strcasecmp(irc_get_raw_nick($target), $nick))
								unset($nick_list[$key]);
						
						$channel->nick_list = $nick_list;
					}
					
					$irc_channel_array[$i] = $channel;
				}
				
				// call appropriate callbacks
				irc_debug("irc_handle_regular_message(): QUIT Message passed on to callback - Code ".IRCCB_ONPART.", $nick!$identd@$host has quit");
				irc_do_callback(IRCCB_ONQUIT, $nick, $identd, $host, "", $text);

				break;
				
		case "KICK":	// KICK message, format:
				// "<origin> KICK <channel> <nick> :<reason>"
				
				// get channel index
				$idx = irc_get_channel_by_name($destination);
				
				// break on invalid channel
				if(!$idx)
					break;
					
				// get channel descriptor
				$channel = $irc_channel_array[$idx - 1];

				// get nick list from channel				
				$nick_list = $channel->nick_list;
				
				// remove nick from channel
				while(list($key, $target) = each($nick_list))
					if(!strcasecmp(irc_get_raw_nick($target), $kick_nick))
						unset($nick_list[$key]);

				$channel->nick_list = $nick_list;
				
				$irc_channel_array[$idx - 1] = $channel;
				
				irc_debug("irc_handle_regular_message(): KICK Message passed on to callback - Code ".IRCCB_ONKICK.", $nick!$identd@$host has kicked $kick_nick from $destination");
				irc_do_callback(IRCCB_ONKICK, $nick, $identd, $host, $destination, $kick_nick);
				
				break;
		
		case "NICK":	// somebody changed his nick
				// the format here is
				// NICK :<new_nick>
				// new_nick is already in $text
				irc_debug("irc_handle_regular_message(): $nick changes his nick to $text, passing on to callback");
				
				// find all related channels and change nickname there
				for($i = 0; $i < IRC_MAX_CHANNELS; $i++)
				{
					$channel = $irc_channel_array[$i];
					
					if($channel->name != "")
					{
						// this channel slot is in use, check if nick
						// is in nick list
						$nick_list = $channel->nick_list;
						
						while(list($key, $target) = each($nick_list))
						{
							if(!strcasecmp($target, $nick))
							{
								$nick_list[$key] = $text;
							}
							else
							if(!strcasecmp($target, ("@".$nick)))
							{
								$nick_list[$key] = "@".$text;
							}
							else
							if(!strcasecmp($target, ("+".$nick)))
							{
								$nick_list[$key] = "+".$text;
							}
						}
						
						$channel->nick_list = $nick_list;
					}
					
					$irc_channel_array[$i] = $channel;
				}
				
				irc_do_callback(IRCCB_ONNICKCHANGE, $nick, $identd, $host, "", $text);
				
				break;

		case "TOPIC":	// topic change
				// the format here is
				// TOPIC <channel> :<topic>
				irc_debug("irc_handle_regular_message(): $nick changes the topic in $destination to $text, passing on to callback");

				irc_do_callback(IRCCB_ONTOPICCHANGE, $nick, $identd, $host, $destination, $text);

				break;

		case "KILL":	// an IRCOp killed us
				// extract kill message
				$text = strstr($text, "(");
				$text = substr($text, 1, strlen($text) - 2);
				irc_debug("irc_handle_regular_message(): $nick killed us from the network because of \"$text\"");

				irc_do_callback(IRCCB_ONKILL, $nick, $identd, $host, $irc_nick, $text);

				$irc_connected = 0;

				break;

		default:	// unknown message type
				irc_debug("irc_handle_regular_message(): Unhandled message \"$msg\" \"$command\"");
				break;
	}

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_idle($break_condition)
//
///////////////////////////////////////////////////////////////////////////////
//
// This function implements something like a main loop, it does not return
// until a break condition is encountered. After the break condition is
// encountered, the function will return. While looping, this function
// will call the various callbacks that can be set for each event.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$break_condition - server message number to break on
//
///////////////////////////////////////////////////////////////////////////////
//
// Global references:
//	$irc_connected, $irc_server, $irc_nick, $irc_userhost
//
///////////////////////////////////////////////////////////////////////////////

function irc_idle($break_condition)
{
	global $irc_connected, $irc_server, $irc_nick, $irc_userhost;

	$break_flag = 0;

	irc_debug("irc_idle(): New idle loop waiting for $break_condition");

	while(!$break_flag && $irc_connected)
	{
		$msg = irc_get_message();
		irc_debug($msg);

		// we got a message, it is in the following format:
		// ":<origin> <command> <destination> :<parameter>"
		// however, server PINGs and NOTICEs are different:
		// "PING :<origin>"
		// "NOTICE <destination> :<message>"
		
		$match = explode(" ", $msg);
		
		switch($match[0])
		{
			case "PING":	// format: "PING :<origin>"
					// extract origin of ping
					$origin = substr($match[1], 1);
					// send PONG back
					irc_put_message("PONG $origin");
					irc_debug("irc_idle(): PING? PONG! ($origin)");
					
					break;
					
			case "NOTICE":	// this is a notice, don't handle it right now
					irc_debug("irc_idle(): Skipping server notice \"$msg\"");
					break;
		
			default:	// break down the message using a regular expression:
					// this one filters out the origin (nick or userhost),
					// and leaves the rest of the message untouched
					$origin = substr($msg, 1, strpos($msg, " ") - 1);
					$rest = substr(strchr($msg, " "), 1);

					// depending on the origin, we have to handle the server messages differently
					switch($origin)
					{
						case $irc_server:	// this is a message from the server directly
									$break_flag = irc_handle_server_message($rest, $break_condition);
									break;

//						case $irc_userhost:
//						case $irc_nick:		// message from ourselves
//									irc_handle_own_message($rest);
//									break;
				
						default:		// message from someone else
									irc_handle_regular_message($origin, $rest);
									break;
					}
		}
		
	}

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_get_color_idx($text, $pos)
//
///////////////////////////////////////////////////////////////////////////////
//
// Tool function for irc_color_decode():
//
// Returns the color index at a given position in the specified string. If no
// color index is present at that position, it returns an empty string.
//
// This function has originally been created by Jirka Kysela (aceman@qwerty.cz)
// and has been adapted for phpIRC by Till Gerken.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$text - string to search in
//	$pos - position to start searching at
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	color code as string
//
///////////////////////////////////////////////////////////////////////////////

function irc_get_color_idx($text, $pos)
{

	// determine the length of the color code
	if((ord($text[$pos]) >= 48) && (ord($text[$pos]) <= 57))
	{
		$cchars = 1;

		if(($pos + 1) < strlen($text))
			if((ord($text[$pos + 1]) >= 48) && (ord($text[$pos + 1]) <= 57))
				$cchars = 2;
	}
	else
	{
		// no color code present at given position
		$cchars = 0;
	}

	return(substr($text, $pos, $cchars));

}

///////////////////////////////////////////////////////////////////////////////
//
// function irc_color_decode($text, $usebgcolor = 0)
//
///////////////////////////////////////////////////////////////////////////////
//
// Decodes IRC color codes (as used by mIRC, Pirch and various other IRC
// clients) to valid HTML
//
// This function has originally been created by Jirka Kysela (aceman@qwerty.cz)
// and has been adapted for phpIRC by Till Gerken.
//
///////////////////////////////////////////////////////////////////////////////
//
// Parameters:
//	$text - string to decode
//	$usebgcolor - 1=display background color [optional]
//
///////////////////////////////////////////////////////////////////////////////
//
// Return value:
//	Decoded colored string as HTML
//
///////////////////////////////////////////////////////////////////////////////

function irc_color_decode($text, $usebgcolor = 0)
{

	// this colors palette is equal to mirc colors palette
	$color[0] ="#ffffff";			// white
	$color[1] ="#000000";			// black
	$color[2] ="#00007f";			// blue
	$color[3] ="#007f00";			// green
	$color[4] ="#ff0000";			// light red
	$color[5] ="#7f0000";			// red
	$color[6] ="#9f009f";			// magenta
	$color[7] ="#ff7f00";			// orange
	$color[8] ="#ffff00";			// yellow
	$color[9] ="#00ff00";			// light green (lime)
	$color[10]="#006464";			// cyan
	$color[11]="#00ffff";			// light cyan (aqua)
	$color[12]="#0000ff";			// light blue
	$color[13]="#c200c2";			// light magenta (pink)
	$color[14]="#7a7a7a";			// grey
	$color[15]="#a6a6a6";			// light grey (silver)

	$text = str_replace(" ", "&nbsp;", $text);

	$text = explode("\3", $text);
	$ctext = $text[0];

	if(count($text) > 1)
	{
		for($i = 1; $i < count($text); $i++)
		{
			if ($text[$i])
			{
				$ccharslen = strlen($showfgcolor = irc_get_color_idx($text[$i], 0));

				if(($ccharslen == 0) || (($showfgcolor = (integer)$showfgcolor) > (count($color) - 1)))
					$showfgcolor = -1;

				if(($text[$i][$ccharslen] == ",") && ($usebgcolor))
				{
					if(strlen($showbgcolor = irc_get_color_idx($text[$i], $ccharslen + 1)))
					{
						$ccharslen += 1 + strlen($showbgcolor);

						if(($showbgcolor = (integer)$showbgcolor) > (count($color) - 1))
						{
							$showbgcolor = -1;
						}

					}
				 	else
				 	{
				 		$showbgcolor = -1;
				 	}

				}
				else
				{
					$showbgcolor = -1;
				}

				if($showfgcolor != -1)
					$ctext .= "<font color=\"$color[$showfgcolor]\">";

				$ctext .= substr($text[$i], $ccharslen, strlen($text[$i]) - $ccharslen);

				if ($showfgcolor != -1)
					$ctext.="</font>";
			}
		}
	}

	// bold text
	$text = explode("\2", $ctext);
	$ctext = $text[0];

	if(count($text) > 1)
	{
		for($i = 1; $i < count($text); $i++)
			if($text[$i])
				$ctext .= "<b>$text[$i]</b>";
	}

	// text underlining
	$text = explode("\37", $ctext);
	$ctext = $text[0];

	if(count($text) > 1)
	{
		for($i = 1; $i < count($text); $i++)
			if($text[$i])
				$ctext .= "<u>$text[$i]</u>";
	}

	// italics
	$text = explode("\26", $ctext);
	$ctext = $text[0];

	if(count($text) > 1)
	{
		for($i = 1; $i < count($text); $i++)
			if($text[$i])
				$ctext .= "<i>$text[$i]</i>";
	}

	// symbol
	$text = explode("\22", $ctext);
	$ctext = $text[0];

	if(count($text) > 1)
	{
		for($i = 1; $i < count($text); $i++)
			if($text[$i])
				$ctext .= "<font face=\"symbol\">$text[$i]</font>";
	}

	// fixed
	$text = explode("\21", $ctext);
	$ctext = $text[0];

	if(count($text) > 1)
	{
		for($i = 1; $i < count($text); $i++)
			if($text[$i])
				$ctext .= "<font face=\"fixedsys\">$text[$i]</font>";
	}

	return("<font face=\"fixedsys\">".$ctext."</font>");


}

///////////////////////////////////////////////////////////////////////////////

?>