<?php if (!defined ("_SQLSTORAGE_CLASS_")):

//# ==================================================
//# Softerra PHP developer's library
//# Copyright (c) Softerra LLC 2000, 2001
//# 
//# This library is free software; you can redistribute it and/or
//# modify it under the terms of the GNU Lesser General Public
//# License as published by the Free Software Foundation; either
//# version 2.1 of the License, or (at your option) any later version.
//# 
//# Contacts:
//# http://www.softerra.com
//# mailto:contacts@softerra.com
//# ==================================================

// ==========================================================
//            SQL Storage Abstraction Layer Class
//                      Version 1.5
// ==========================================================
//  simple wrapper class for SQL libraries "SAL" and "sqlcompose.lib.php"
//                       
// DECLARED METHODS:
//
// 	connect	($host, $user, $password, $name)
// 	query	($query)
// 	select 	($tables, $fields="*", $conditions=0)
// 	update 	($table, $values, $conditions=0)
// 	insert 	($table, $values)
// 	delete 	($table, $conditions=0)
// 	select_exists 	($tables, $conditions=0)
// 	select_count 	($tables, $conditions=0)
// 	select_record 	($tables, $fields, $conditions=0)
// 	select_row 	($tables, $fields, $conditions=0)
// 	select_column 	($tables, $fields, $conditions=0)
// 	select_result 	($tables, $fields, $conditions=0, $format=fmtArr)
//	affected_rows ()
//
// ==========================================================
// ==========================================================
 
/**
 * File name.
 *
 * @const	_SQLSTORAGE_CLASS_ 
 * @access	private
 */
define (_SQLSTORAGE_CLASS_, "sqlstorage.class.php");

require ($lib_dir . "lib.cfg.php");
require ($lib_dir . "corban.lib.php");
require ($lib_dir . "sqlcompose.lib.php");
require ($lib_dir . "sqlsearch.lib.php");
require ($lib_dir . "sal/" . $salServer . ".fun.php");

/**
 * Error message.
 *
 * @const	strErrorConnect 
 * @access	private
 */
define (strErrorConnect, "Fatal: Can't connect to database.\n");
/**
 * Error message.
 *
 * @const	strErrorSelect 
 * @access	private
 */
define (strErrorSelect, "Fatal: Can't select from database.\n");

/**
 * Row format type.
 *
 * @const	fmtRow 
 * @access	private
 */
define (fmtRow, 1);
/**
 * Array format type.
 *
 * @const	fmtArr
 * @access	private
 */
define (fmtArr, 2);
/**
 * Object format type.
 *
 * @const	fmtObj
 * @access	private
 */
define (fmtObj, 3);

/**
 * Print error message.
 *
 * @param	string	$method
 * @return	boolean
 * @access	private
 */
function sal_unsupported_method ($method) 
{
	$msg = "SAL error!\\nMethod '$method' unsupported by " . strtoupper ($GLOBALS["salServer"]) . " server.";
	echo "<script language=\"JavaScript\">";
	echo "alert (\"$msg\")";
	echo "</script>";
	return false;
}

/**
 * SQL Storage Abstraction Layer Class.
 *
 * Simple wrapper class for SQL.
 * @author Softerra LLC <contacts@softerra.com>
 * @access	public
 * @version	1.5
 * @copyright	(C) 2000-2001 Softerra LLC
 */
class sqlStorage 
{
	
/**
 * Debug flag.
 *
 * @var	boolean
 */
	var $debug;
/**
 * Debug message.
 *
 * @var	string
 */
	var $debugmessage;
/**
 * Connection link ID.
 *
 * @var	integer
 */
	var $connection;
/**
 * Result data link ID.
 *
 * @var	integer
 */
	var $queryresult;
/**
 * DB link ID.
 *
 * @var	integer
 */
	var $database;
/**
 * Query counter.
 *
 * @var	integer
 */
	var $query_count;
	
/**
 * Constructor of SQL Storage Class.
 *
 * 
 * @access	public
 */
	function sqlStorage () 
	{
		$this->query_count = 0;
	}

/**
 * Get SQL server connect to.
 *
 * @return	string Type of the SQL server eg. "MySQL"
 * @access	public
 */
	function getServer () 
	{
		return $GLOBALS["salServer"];
	}

/**
 * Turn on a SQL debugger.
 *
 * @access	public
 */
	function debugOn () 
	{
		$this->debug = true;
	}
	
/**
 * Turn off a SQL debugger.
 *
 * @access	public
 */
	function debugOff () 
	{
		$this->debug = false;
	}
	
/**
 * Open a connection to a SQL Server.
 *
 * Stop HTML output and display an error message on failure. 
 *
 * @param 	string  $host The hostname string can also include a port number. eg. "hostname:port". 
 * @param	string  $user User name
 * @param	string  $password Password
 * @param	string  $name DB name 
 * @access	public
 * @see close()
 */
	function connect ($host, $user, $password, $name) 
	{
		$this->connection = sql_connect ($host, $user, $password, $name) or die (strErrorConnect . $this->error ());
		$this->database = $name;
	}
	
/**
 * Close a SQL connection.
 *
 * This isn't usually necessary, as non-persistent open links are automatically closed
 * at the end of the script's execution.  
 *
 * @access	public
 * @see 	connect()
 */
	function close () 
	{
		sql_close ($this->connection);
	}
	
/**
 * Select a SQL database.
 *
 * Set the current active database on the server.
 * 
 *
 * @param	string 	$database DB name
 * @access	public
 * @see 	connect()
 */
	function select_db ($database) 
	{
		if (sql_select_db ($database))	$this->database = $database;
	}

/**
 * List databases available on a SQL server .
 *
 * Return an array of the database names available from the current sql daemon.
 * 
 *
 * @access	public
 * @return 	array
 * @see 	list_tables()
 * @see 	list_fields()
 */
	function list_dbs () 
	{
		return sql_list_dbs ($this->connection);
	}

/**
 * List tables in a SQL database.
 *
 * Take a database name and returns an array of table names.
 * 
 *
 * @param	string	$database DB Name
 * @access	public
 * @return 	array
 * @see 	list_dbs()
 * @see 	list_fields()
 */
	function list_tables ($database="") 
	{
		if (!$database) $database = $this->database;
		return sql_list_tables ($database, $this->connection);
	}

/**
 * List SQL result fields.
 *
 * Take a table name and returns an array of field names.
 * 
 *
 * @param	string	$table 	Table Name
 * @param	string	$database DB Name (optional)
 * @access	public
 * @return 	array
 * @see 	list_dbs()
 * @see 	list_tables()
 */
	function list_fields ($table, $database="") 
	{
		if (!$database) $database = $this->database;
		return $this->queryresult = sql_list_fields ($database, $table, $this->connection);
	}

/**
 * Return the text of the error message from previous SQL operation.
 *
 * Return the empty string if no error occurred.
 * 
 *
 * @access	public
 * @return 	string
 */
	function error () 
	{
		return sql_error ($this->connection);
	}

/**
 * Free result memory.
 *
 * Only need to be called if you are worried about using too much memory 
 * while your script is running. All associated result memory 
 * for the specified result identifier will automatically be freed.
 * 
 *
 * @access	public
 * @return 	integer
 */
	function free_result () 
	{
		return sql_free_result ($this->queryresult);
	}

/**
 * Send a SQL query.
 *
 * The query string should not end with a semicolon.
 * Return TRUE (non-zero) or FALSE to indicate whether or not the query succeeded.
 * 
 * @param	string	$query
 * @access	public
 * @return 	integer
 * @see 	select()
 * @see 	update()
 * @see 	insert()
 * @see 	delete()
 */
	function query ($query) 
	{
		if ($this->debug) 
		{
			cd ($query, "query", $this->debugmessage);
			ci ("query");
		}
		$this->queryresult = sql_query ($query, $this->connection);
		$this->query_count++;
		if ($this->debug) 
		{
			cs ("query", "Query execution time (sec.) ");
			if (!$this->queryresult) cd ($this->error (), "error", $this->debugmessage);
		}
		return $this->queryresult;
	}
	
/**
 * Get number of affected rows in previous SQL operation.
 *
 * Return the number of rows affected by the last INSERT, UPDATE or DELETE query.
 * If the last query was a DELETE query with no WHERE clause, 
 * all of the records will have been deleted from the table but this function will return zero. 
 * This method is not effective for SELECT statements, only on statements which modify records.
 *
 * @return	integer	
 * @access	public
 * @see 	num_rows()
*/
	function affected_rows () 
	{
		return sql_affected_rows ($this->queryresult);
	}
	
/**
 * Send a SQL SELECT query.
 *
 * Return TRUE (non-zero) or FALSE to indicate whether or not the query succeeded.
 * 
 * @param	string	$tables Table name(s), can be also an array.
 * @param	string	$fields Field name(s), can be also an array. If no specified select all fields.
 * @param	string	$conditions Condition, can be also an array.
 * @access	public
 * @return 	integer
 * @see 	query()
 * @see 	select_count()
 * @see 	select_exists()
 * @see 	select_record()
 * @see 	select_row()
 * @see 	select_column()
 * @see 	select_result()
 */
	function select ($tables, $fields="*", $conditions="") 
	{
		$query = composeSelectQuery ($tables, $fields, $conditions);
		return $this->query ($query) or die (strErrorSelect . $this->error ());
	}

/**
 * Send a SQL UPDATE query.
 *
 * Return TRUE (non-zero) or FALSE to indicate whether or not the query succeeded.
 * 
 * @param	string	$table Table name.
 * @param	array	$values An associative array contains field name(s) as key(s).
 * @param	string	$conditions Condition, can be also an array.
 * @access	public
 * @return 	integer
 * @see 	query()
 */
	function update ($table, $values, $conditions="") 
	{
		$query = composeUpdateQuery ($table, $values, $conditions);
		return $this->query ($query);
	}

/**
 * Send a SQL INSERT query.
 *
 * Return TRUE (non-zero) or FALSE to indicate whether or not the query succeeded.
 * 
 * @param	string	$table Table name.
 * @param	array	$values An associative array contains field name(s) as key(s).
 * @access	public
 * @return 	integer
 * @see 	query()
 * @see 	insert_id()
 */
	function insert ($table, $values) 
	{
		$query = composeInsertQuery ($table, $values);
		return $this->query ($query);
	}
	
/**
 * Send a SQL DELETE query.
 *
 * Return TRUE (non-zero) or FALSE to indicate whether or not the query succeeded.
 * 
 * @param	string	$table Table name.
 * @param	string	$conditions Condition, can be also an array.
 * @access	public
 * @return 	integer
 * @see 	query()
 */
	function delete ($table, $conditions="") 
	{
		$query = composeDeleteQuery ($table, $conditions);
		return $this->query ($query);
	}
	
/**
 * Get result data.
 *
 * Return the contents of one cell from a SQL result set. 
 * The field argument can be the field's offset, or the field's name, 
 * or the field's table dot field's name (fieldname.tablename). 
 * If the column name has been aliased ('select foo as bar from...'), 
 * use the alias instead of the column name. 
 * 
 * @param	integer	$row 
 * @param	integer	$field 
 * @access	public
 * @return 	integer
 * @see 	fetch_row()
 * @see 	fetch_array()
 * @see 	fetch_object()
 * @see 	fetch_field()
 * @see 	fetch_fields()
 * @see 	fetch_result()
 * @see 	fetch_rows()
 * @see 	fetch_arrays()
 * @see 	fetch_objects()
 */
	function result ($row=0, $field=0) 
	{
		return sql_result ($this->queryresult, $row, $field);
	}

/**
 * Get number of rows in result.
 *
 * This method is only valid for SELECT statements. 
 * 
 * @access	public
 * @return 	integer
 * @see 	affected_rows()
 * @see 	num_fields()
 */
	function num_rows () 
	{
		return sql_num_rows ($this->queryresult);
	}

/**
 * Get number of fields in result.
 *
 * This method is only valid for SELECT statements. 
 * 
 * @access	public
 * @return 	integer
 * @see 	num_rows()
 */
	function num_fields () 
	{
		return sql_num_fields ($this->queryresult);
	}

/**
 * Get a result row as an enumerated array.
 *
 * Fetch one row of data from the result associated with the specified result identifier.
 * The row is returned as an array. Each result column is stored in an array offset, 
 * starting at offset 0. 
 * 
 * @access	public
 * @return 	array
 * @see 	result()
 * @see 	fetch_array()
 * @see 	fetch_object()
 * @see 	fetch_field()
 * @see 	fetch_fields()
 * @see 	fetch_result()
 * @see 	fetch_rows()
 * @see 	fetch_arrays()
 * @see 	fetch_objects()
 */
	function fetch_row () 
	{
		return sql_fetch_row ($this->queryresult);
	}

/**
 * Fetch a result row as an associative array.
 *
 * In addition to storing the data in the numeric indices of the result array,
 * it also stores the data in associative indices, using the field names as keys. 
 * 
 * @access	public
 * @return 	array
 * @see 	result()
 * @see 	fetch_row()
 * @see 	fetch_object()
 * @see 	fetch_field()
 * @see 	fetch_fields()
 * @see 	fetch_result()
 * @see 	fetch_rows()
 * @see 	fetch_arrays()
 * @see 	fetch_objects()
 */
	function fetch_array () 
	{
		return sql_fetch_array ($this->queryresult);
	}

/**
 * Fetch a result row as an object.
 *
 * Return an object with properties that correspond to the fetched row, 
 * or false if there are no more rows.
 * 
 * @param	integer	$type
 * @access	public
 * @return 	Object 	
 * @see 	result()
 * @see 	fetch_row()
 * @see 	fetch_array()
 * @see 	fetch_field()
 * @see 	fetch_fields()
 * @see 	fetch_result()
 * @see 	fetch_rows()
 * @see 	fetch_arrays()
 * @see 	fetch_objects()
 */
	function fetch_object ($type=0) 
	{
		return sql_fetch_object ($this->queryresult, $type);
	}

/**
 * Get column information from a result and return as an object .
 *
 * Can be used in order to obtain information about fields in a certain query result.
 * If the field offset isn't specified, the next field that wasn't yet retrieved 
 * by fetch_field() is retrieved. 
 * 
 * @param 	integer	$offset
 * @access	public
 * @return 	Object
 * @see 	result()
 * @see 	fetch_row()
 * @see 	fetch_array()
 * @see 	fetch_object()
 * @see 	fetch_fields()
 * @see 	fetch_result()
 * @see 	fetch_rows()
 * @see 	fetch_arrays()
 * @see 	fetch_objects()
 */
	function fetch_field ($offset) 
	{
		return sql_fetch_field ($this->queryresult, $offset);
	}
	
/**
 * Get column information from a result and return as array of objects.
 *
 * Can be used in order to obtain information about fields in a certain query result.
 * 
 * @access	public
 * @see 	result()
 * @see 	fetch_row()
 * @see 	fetch_array()
 * @see 	fetch_object()
 * @see 	fetch_field()
 * @see 	fetch_result()
 * @see 	fetch_rows()
 * @see 	fetch_arrays()
 * @see 	fetch_objects()
 * @return 	array
 */
	function fetch_fields () 
	{
		for ($i=0; $i<$this->num_fields (); $i++)
			$fields[] = $this->fetch_field ($i);
		return $fields;
	}
	
/**
 * Fetch all rows of result data into an multi-dimensional array.
 *
 * Return a multi-dimensional array[row][field] or array[row] of objects depending on $format parameter.
 * 
 * @param 	integer	$format Accept the following constants: fmtRow, fmtArr, fmtObj.
 * @access	public
 * @return 	array
 * @see 	result()
 * @see 	fetch_row()
 * @see 	fetch_array()
 * @see 	fetch_object()
 * @see 	fetch_field()
 * @see 	fetch_fields()
 * @see 	fetch_rows()
 * @see 	fetch_arrays()
 * @see 	fetch_objects()
 */
	function fetch_result ($format=fmtRow) 
	{
		$result = array ();
		for ($i=0; $i<$this->num_rows (); $i++) 
		{
			switch ($format) 
			{
				case fmtRow:
					$result[] = $this->fetch_row ();
				break;
				case fmtArr:
					$result[] = $this->fetch_array ();
				break;
				case fmtObj:
					$result[] = $this->fetch_object ();
				break;
			}
		}
		return $result;
	}
	
/**
 * Fetch all rows of result data into an multi-dimensional array.
 *
 * Return a multi-dimensional array[row][field].
 * 
 * @access	public
 * @return 	array
 * @see 	result()
 * @see 	fetch_row()
 * @see 	fetch_array()
 * @see 	fetch_object()
 * @see 	fetch_field()
 * @see 	fetch_fields()
 * @see 	fetch_result()
 * @see 	fetch_arrays()
 * @see 	fetch_objects()
 */
	function fetch_rows () 
	{
		return $this->fetch_result (fmtRow);
	}

/**
 * Fetch all rows of result data into an multi-dimensional array.
 *
 * Return a multi-dimensional array[row][field].
 * 
 * @access	public
 * @return 	array
 * @see 	result()
 * @see 	fetch_row()
 * @see 	fetch_array()
 * @see 	fetch_object()
 * @see 	fetch_field()
 * @see 	fetch_fields()
 * @see 	fetch_result()
 * @see 	fetch_rows()
 * @see 	fetch_objects()
 */
	function fetch_arrays () 
	{
		return $this->fetch_result (fmtArr);
	}

/**
 * Fetch all rows of result data.
 *
 * Return an array[row] of objects.
 * 
 * @access	public
 * @return 	array
 * @see 	result()
 * @see 	fetch_row()
 * @see 	fetch_array()
 * @see 	fetch_field()
 * @see 	fetch_fields()
 * @see 	fetch_result()
 * @see 	fetch_rows()
 * @see 	fetch_arrays()
 * @see 	fetch_objects()
 */
	function fetch_objects () 
	{
		return $this->fetch_result (fmtObj);
	}
	
/**
 * Get the id generated from the previous INSERT operation.
 *
 * Return the ID generated for an AUTO_INCREMENTED field.
 * 
 * @access	public
 * @return 	integer
 * @see 	insert()
 */
	function insert_id () 
	{
		return sql_insert_id ($this->queryresult, $this->connection);
	}
	
/**
 * Check if the previous SELECT operation returns result data.
 *
 * Return TRUE if the previous SELECT operation returns result data.
 * 
 * @access	public
 * @return 	boolean
 * @see 	queryresult_record()
 * @see 	queryresult_column()
 */
	function queryresult_exists () 
	{
		return ($this->num_rows () > 0);
	}
	
/**
 * Fetch next row and return value of the first field.
 *
 * 
 * @access	public
 * @return 	integer
 * @see 	queryresult_exists()
 * @see 	queryresult_column()
 */
	function queryresult_record () 
	{
		$row = $this->fetch_row ();
		return $row[0];
	}
	
/**
 * Fetch all rows of result data and return values of the first column.
 *
 * 
 * @access	public
 * @return 	array
 * @see 	queryresult_record()
 * @see 	queryresult_exists()
 */
	function queryresult_column () 
	{
		$column = array ();
		while ($row = $this->fetch_row ())
			$column[] = $row[0];
		return $column;
	}

/**
 * Send a SELECT query and check if result data is not empty.
 *
 * @param	string	$tables Table name(s), can be also an array.
 * @param	string	$conditions Condition, can be also an array.
 * @access	public
 * @return 	boolean
 * @see 	select()
 * @see 	select_count()
 * @see 	select_record()
 * @see 	select_row()
 * @see 	select_column()
 * @see 	select_result()
 */
	function select_exists ($tables, $conditions="") 
	{
		$this->select ($tables, "*", $conditions);
		return $this->queryresult_exists ();
	}
	
/**
 * Prepare and send a SELECT query to count number of rows for the specified conditions.
 *
 * @param	string	$tables Table name(s), can be also an array.
 * @param	string	$conditions Condition, can be also an array.
 * @access	public
 * @return 	integer
 * @see 	select()
 * @see 	select_exists()
 * @see 	select_record()
 * @see 	select_row()
 * @see 	select_column()
 * @see 	select_result()
 */
	function select_count ($tables, $conditions="") 
	{
		$this->select ($tables, "count(*) as TOTAL", $conditions);
		if (substr_count($conditions, "group by")) return intval ($this->num_rows());
		else return intval ($this->queryresult_record ());
	}

/**
 * Send a SELECT query and return value of the first row and the first field.
 *
 * @param	string	$tables Table name(s), can be also an array.
 * @param	string	$fields Field name(s), can be also an array. If no specified select all fields.
 * @param	string	$conditions Condition, can be also an array.
 * @access	public
 * @return 	integer
 * @see 	select()
 * @see 	select_count()
 * @see 	select_exists()
 * @see 	select_row()
 * @see 	select_column()
 * @see 	select_result()
 */
	function select_record ($tables, $fields, $conditions="") 
	{
		$this->select ($tables, $fields, $conditions);
		return $this->queryresult_record ();
	}
	
/**
 * Send a SELECT query and return the first result row as an associative array.
 *
 * @param	string	$tables Table name(s), can be also an array.
 * @param	string	$fields Field name(s), can be also an array. If no specified select all fields.
 * @param	string	$conditions Condition, can be also an array.
 * @access	public
 * @return 	array
 * @see 	select()
 * @see 	select_count()
 * @see 	select_exists()
 * @see 	select_record()
 * @see 	select_column()
 * @see 	select_result()
 */
	function select_row ($tables, $fields, $conditions="") 
	{
		$this->select ($tables, $fields, $conditions);
		return $this->fetch_array ();
	}
	
/**
 * Send a SELECT query and return values of the first column.
 *
 * @param	string	$tables Table name(s), can be also an array.
 * @param	string	$fields Field name(s), can be also an array. If no specified select all fields.
 * @param	string	$conditions Condition, can be also an array.
 * @access	public
 * @return 	array
 * @see 	select()
 * @see 	select_count()
 * @see 	select_exists()
 * @see 	select_record()
 * @see 	select_row()
 * @see 	select_result()
 */
	function select_column ($tables, $fields, $conditions="") 
	{
		$this->select ($tables, $fields, $conditions);
		return $this->queryresult_column ();
	}

/**
 * Send a SELECT query and return all rows of result data into an multi-dimensional array.
 *
 * @param	string	$tables Table name(s), can be also an array.
 * @param	string	$fields Field name(s), can be also an array. If no specified select all fields.
 * @param	string	$conditions Condition, can be also an array.
 * @param 	integer	$format Accept the following constants: fmtRow, fmtArr, fmtObj
 * @access	public
 * @return 	array
 * @see 	select()
 * @see 	select_count()
 * @see 	select_exists()
 * @see 	select_record()
 * @see 	select_row()
 * @see 	select_column()
 */
	function select_result ($tables, $fields, $conditions="", $format=fmtArr) 
	{
		$this->select ($tables, $fields, $conditions);
		return $this->fetch_result ($format);
	}
	
/**
 * Prepare and send a SELECT query according specified espression.
 *
 * Return all rows of result data into an multi-dimensional array.
 *
 * @param	string	$expression 	Expression.
 * @param	string	$table 	Table name.
 * @param	string	$field 	Field name.
 * @param	string	$fields Field name(s), can be also an array. If no specified select all fields.
 * @param 	integer	$type 	Accept the following constants: SQL_PATTERN, REGEXP_PATTERN
 * @param 	integer	$limit 	Limit of result data. 
 * @access	public
 * @return 	array
 */
	function search ($expression, $table, $field="", $fields="*", $type=SQL_PATTERN, $limit=0) 
	{
		if ($field) targetfield ($field);
		$conditions = expression2conditions ($expression, $type);
		$conditions = standalone ($conditions);
		if ($limit) 
		{
			$cnt = $this->select_count ($table, $conditions);
			$conditions .= " limit $limit[FIRST], $limit[NUM]";
		}
		$this->select ($table, $fields, $conditions);
		if (!$limit) $cnt = $this->num_rows ();
		$result = $this->fetch_result (fmtArr);
		$result["COUNT"] = $cnt;
		return $result;
	}
}

endif ?>