<?php
//
// +------------------------------------------------------------------------+
// | phpDocumentor                                                          |
// +------------------------------------------------------------------------+
// | Copyright (c) 2000-2003 Joshua Eichorn, Gregory Beaver                 |
// | Email         jeichorn@phpdoc.org, cellog@phpdoc.org                   |
// | Web           http://www.phpdoc.org                                    |
// | Mirror        http://phpdocu.sourceforge.net/                          |
// | PEAR          http://pear.php.net/package-info.php?pacid=137           |
// +------------------------------------------------------------------------+
// | This source file is subject to version 3.00 of the PHP License,        |
// | that is available at http://www.php.net/license/3_0.txt.               |
// | If you did not receive a copy of the PHP license and are unable to     |
// | obtain it through the world-wide-web, please send a note to            |
// | license@php.net so we can mail you a copy immediately.                 |
// +------------------------------------------------------------------------+
//

/**
 * All abstract representations of inline tags are in this file
 * @package phpDocumentor
 * @subpackage InlineTags
 * @since separate file since version 1.2
 * @version $Id: InlineTags.inc,v 1.21.2.3 2003/06/03 05:44:52 CelloG Exp $
 */
/**
 * Use this element to represent an {@}inline tag} like {@}link}
 * @see parserStringWithInlineTags
 * @package phpDocumentor
 * @subpackage InlineTags
 * @author Greg Beaver <cellog@users.sourceforge.net>
 * @since 1.0rc1
 * @version $Revision: 1.21.2.3 $
 * @tutorial inlinetags.pkg
 */
class parserInlineTag extends parserBase
{
    /**
     * Element type
     *
     * Type is used by many functions to skip the hassle of
     *
     * <code>
     * if get_class($blah) == 'parserBlah'
     * </code>
     * always "inlinetag"
     * @var string
     */
    var $type = 'inlinetag';
    /**
     * the name of the inline tag (like link)
     * @var string
     */
    var $inlinetype = '';
    
    /**
     * @param string $type tag type (example: link)
     * @param string $value tag value (example: what to link to)
     */
    function parserInlineTag($type,$value)
    {
        $this->inlinetype = $type;
        $this->value = trim($value);
    }
    
    /**
     * @return integer length of the tag
     */
    function Strlen()
    {
        return strlen($this->value);
    }
    
    /**
     * @return string always '', used by {@link Parser::handleDocBlock()} to
     *                calculate the short description of a DocBlock
     * @see parserStringWithInlineTags::getString()
     * @see parserStringWithInlineTags::trimmedStrlen()
     */
    function getString()
    {
        return '';
    }
}

/**
 * represents inline links
 * @tutorial tags.inlinelink.pkg
 * @package phpDocumentor
 * @subpackage InlineTags
 * @author Greg Beaver <cellog@users.sourceforge.net>
 * @since 1.0rc1
 */
class parserLinkInlineTag extends parserInlineTag
{
    /**
     * text to display in the link, can be different from the link for standard
     * links like websites
     * @var string
     */
    var $linktext = '';
    
    /**
     * @param string $link stored in $value, see {@link parserBase::$value}
     * @param string $text see {@link $linktext}
     */
    function parserLinkInlineTag($link,$text)
    {
        if ($link{strlen($link) - 1} == ',')
        {
            $text = explode(',',$text);
            $newtext = '';
            if (count($text) > 1)
            {
                $addlink = array_shift($text);
                $newtext = join($text);
                $text = $addlink;
            } else $text = $text[0];
            $link = $link . $text;
            $text = $newtext;
        }
        parserInlineTag::parserInlineTag('link',$link);
        $this->linktext = trim($text);
    }
    
    /**
     * @param Converter converter used to change the abstract link into text for
     *                  display
     * @return false|string returns the converted link or false if not converted
     *                      successfully
     */
    function Convert(&$c)
    {
        if (strpos($this->value,'://') || (strpos($this->value,'mailto:') === 0))
        {
            return $c->returnLink($this->value,htmlspecialchars($this->linktext));
        } else
        {
            $descrip = false;
            if (strpos($this->value,',') === false)
            {
                if (strpos(trim($this->value),' '))
                {
                    $v = split(' ',trim($this->value));
                    $value = $c->getLink($v[0]);
                    array_shift($v);
                    $descrip = join($v,' ');
                } else
                $value = $c->getLink($this->value);
            } else
            {
                $vals = split(',',$this->value);
                $descrip = array();
                foreach($vals as $val)
                {
                    $val = trim($val);
                    if (strpos($val,' '))
                    {
                        $v = split(' ',$val);
                        $value[] = $c->getLink($v[0]);
                        array_shift($v);
                        $descrip[] = join($v,' ');
                    } else
                    {
                        $value[] = $c->getLink($val);
                        $descrip[] = false;
                    }
                }
            }
            if (is_string($value))
            {
                // feature 564991
                if (strpos($value,'://'))
                {
                    // php function
                    return $c->returnLink($value,str_replace('PHP_MANUAL#','',$this->value));
                }
                return $value;
            }
            if (!$descrip) $descrip = $c->type_adjust($this->linktext);
            if (is_object($value)) return $c->returnSee($value,$descrip);
            // getLink parsed a comma-delimited list of linked thingies, add the commas back in
            if (is_array($value))
            {
                $a = '';
                foreach($value as $i => $bub)
                {
                    if (!empty($a)) $a .= ', ';
                    if (is_string($value[$i]))
                    {
                        // feature 564991
                        if (strpos($value[$i],'://'))
                        {
                            // php function
                            $a .= $c->returnLink($value[$i],str_replace('PHP_MANUAL#','',$vals[$i]));
                        } else
                        $a .= $value[$i];
                    }
                    if (is_object($value[$i])) $a .= $c->returnSee($value[$i],$descrip[$i]);
                }
                return $a;
            }
            return false;
        }
    }
}

/**
 * Represents inline links to external tutorial documentation
 * @tutorial tags.inlinetutorial.pkg
 * @package phpDocumentor
 * @subpackage InlineTags
 */
class parserTutorialInlineTag extends parserLinkInlineTag
{
    /**
     * @param string $link stored in $value, see {@link parserBase::$value}
     * @param string $text see {@link $linktext}
     */
    function parserTutorialInlineTag($link,$text)
    {
        parserInlineTag::parserInlineTag('tutorial',$link);
        $this->linktext = trim($text);
    }

    /**
     * @param Converter converter used to change the abstract link into text for display
     * @return mixed returns the converted link or false if not converted successfully
     */
    function Convert(&$c)
    {
        $descrip = false;
        if (strpos($this->value,',') === false)
        {
            if (strpos(trim($this->value),' '))
            {
                $v = split(' ',trim($this->value));
                $value = $c->getTutorialLink($v[0]);
                array_shift($v);
                $descrip = join($v,' ');
            } else $value = $c->getTutorialLink($this->value);
        } else
        {
            $vals = split(',',$this->value);
            $descrip = array();
            foreach($vals as $val)
            {
                $val = trim($val);
                if (strpos($val,' '))
                {
                    $v = split(' ',$val);
                    $value[] = $c->getTutorialLink($v[0]);
                    array_shift($v);
                    $descrip[] = join($v,' ');
                } else
                {
                    $value[] = $c->getTutorialLink($val);
                    $descrip[] = false;
                }
            }
        }
        if (is_string($value))
        {
            return $value;
        }
        if (is_object($value)) return $c->returnSee($value,$descrip);
        // getLink parsed a comma-delimited list of linked thingies, add the commas back in
        if (is_array($value))
        {
            $a = '';
            foreach($value as $i => $bub)
            {
                if (!empty($a)) $a .= ', ';
                if (is_string($value[$i]))
                {
                    $a .= $value[$i];
                }
                if (is_object($value[$i])) $a .= $c->returnSee($value[$i],$descrip[$i]);
            }
            return $a;
        }
        return false;
    }
}

/**
 * represents inline source tag, used for function/method source
 * @tutorial tags.inlinesource.pkg
 * @package phpDocumentor
 * @subpackage InlineTags
 */
class parserSourceInlineTag extends parserInlineTag
{
    /**
     * always 'source'
     * @var string
     */
    var $inlinetype = 'source';
    /**
     * First line of source code to display
     * @var integer
     * @see $end
     */
    var $start = 1;
    /**
     * Last line to display
     * @var '*'|integer If '*' then the whole source will be used, otherwise
     *                  the {@link $start} to $end line numbers will be displayed
     */
    var $end = '*';
    /**
     * tokenized source organized by line numbers for php 4.3.0+, the old
     * {@}source} tag used a string
     * @var string|array
     */
    var $source = false;
    /**#@+ @access private */
    /** @var string|false */
    var $_class;
    /**#@-*/
    /**
     * @param string format "start [end]" where start and end are line numbers
     *               with the end line number optional
     */
    function parserSourceInlineTag($value)
    {
        parserInlineTag::parserInlineTag('source','');
        preg_match('/^([0-9]*)\W([0-9]*)$/',trim($value), $match);
        if (!count($match))
        {
            preg_match('/^([0-9]*)$/',trim($value),$match);
            if (count($match))
            {
                $this->start = (int) $match[1];
            }
        } else
        {
            $this->start = (int) $match[1];
            $this->end = (int) $match[2];
        }
    }
    
    /**
     * only used to determine blank lines.  {@}source} will not be blank,
     * probably
     */
    function Strlen()
    {
        return 1;
    }
    
    function getString()
    {
        return '{@source}';
    }
    
    /**
     * @param string|array source code
     * @param boolean in php 4.3.0, if this is a method this will be true
     * @param string class name if this is a method
     */
    function setSource($source, $class = false)
    {
        if (is_array($source))
        {
            $this->_class = $class;
            $this->source = $source;
        } else
        {
            $source = strstr($source,'function');
            $pos = strrpos($source,'}');
            $this->source = substr($source,0,$pos + 1);
        }
    }
    
    /**
     * @uses stringConvert() in PHP 4.2.3-, this method is used to convert
     * @uses arrayConvert() in PHP 4.3.0+, this method is used to convert
     * @param Converter
     */
    function Convert(&$c)
    {
        if (is_string($this->source)) return $this->stringConvert($c);
        return $this->arrayConvert($c);
    }
    
    /**
     * @param Converter
     * @uses phpDocumentor_HighlightParser Parses the tokenized source
     */
    function arrayConvert(&$c)
    {
        $source = $this->source;
        if ($this->end != '*')
        {
            $source = array_slice($this->source,0,$this->end + $this->start - 1);
        }
        $parser = new phpDocumentor_HighlightParser;
        $return = '';
        $start = $this->start - 1;
        if ($start < 0) $start = 0;
        return $parser->parse($source, $c, true, $this->_class, $start);
    }
    
    /**
     * @param Converter
     * @uses Converter::unmangle() remove the extraneous stuff from
     *                             {@link highlight_string()}
     * @deprecated in favor of PHP 4.3.0+ {@link arrayConvert()}
     */
    function stringConvert(&$c)
    {
        $source = highlight_string('<?php '.$this->source.' ?>', true);
        $source = '<code>'.substr($source,strlen('<code><font color="#000000">
<font color="#0000CC">&lt;?php&nbsp;</font>') - 1);
        $source = str_replace('}&nbsp;</font><font color="#0000CC">?&gt;</font>','}</font></code>',$source);
        if ($this->start || ($this->end != '*'))
        {
            $source = explode('<br />',$source);
            $start = $this->start;
            if ($this->end != '*')
            {
                $source = array_slice($source,$start - 1,$this->end - $start + 1);
            } else
            {
                $source = array_slice($source,$start - 1);
            }
            $source = implode($source,'<br />');
            if ($start > 0) $source = "<code>$source";
            if ($this->end != '*') $source = "$source</code>";
        }
        $source = $c->unmangle($source,$this->source);
        return $source;
    }
}

/**
 * Represents the inheritdoc inline tag, used by classes/methods/vars to inherit
 * documentation from the parent class if possible
 * @tutorial tags.inlineinheritdoc.pkg
 * @package phpDocumentor
 * @subpackage InlineTags
 */
class parserInheritdocInlineTag extends parserInlineTag
{
    /**
     * always 'inheritdoc'
     * @var string
     */
    var $inlinetype = 'inheritdoc';
    
    /**
     * Does nothing, overrides parent constructor
     */
    function parserInheritdocInlineTag()
    {
    }
    
    function Convert()
    {
        addWarning(PDERROR_INHERITDOC_DONT_WORK_HERE);
        return '';
    }
}

/**
 * Represents the inline {@}id} tag for tutorials
 * @tutorial tags.inlineid.pkg
 * @package phpDocumentor
 * @subpackage InlineTags
 */
class parserIdInlineTag extends parserInlineTag
{
    /**
     * always 'id'
     * @var string
     */
    var $inlinetype = 'id';
    /**
     * package of the {@}id}
     * @var string
     */
    var $package = 'default';
    /**
     * category of the {@}id}
     * @var string
     */
    var $category = 'default';
    /**
     * subpackage of the {@}id}
     * @var string
     */
    var $subpackage = '';
    /**
     * full name of the tutorial
     * @var string
     */
    var $tutorial;
    /**
     * section/subsection name
     * @var string
     */
    var $id;
    
    /**
     * @param string package name
     * @param string subpackage name
     * @param string tutorial name
     * @param string section/subsection name
     * @param string category name
     */
    function parserIdInlineTag($category,$package,$subpackage,$tutorial,$id = false)
    {
        $this->package = $package;
        $this->subpackage = $subpackage;
        $this->tutorial = $tutorial;
        $this->id = $id;
        $this->category = $category;
    }
    
    /**
     * @param Converter
     * @uses Converter::getTutorialId() retrieve converter-specific ID
     */
    function Convert(&$c)
    {
        if (!$this->id) return '';
        return $c->getTutorialId($this->package,$this->subpackage,$this->tutorial,$this->id,$this->category);
    }
}

/**
 * Represents {@}toc} for table of contents generation in tutorials
 * @tutorial tags.inlinetoc.pkg
 * @package phpDocumentor
 * @subpackage InlineTags
 */
class parserTocInlineTag extends parserInlineTag
{
    /**
     * always 'toc'
     * @var string
     */
    var $inlinetype = 'toc';
    /**
     * @var array format:
     * <pre>
     * array(array('tagname' => section,
     *             'link' => returnsee link,
     *             'id' => anchor name,
     *             'title' => from title tag),...)
     * </pre>
     * @access private
     */
    var $_toc = false;
    /**
     * full path to tutorial, used in conversion
     * @var string
     * @access private
     */
    var $_path = false;

    function parserTocInlineTag()
    {
        parent::parserInlineTag('toc','');
    }
    
    /**
     * @param array format:
     * <pre>
     * array(array('tag' => {@link parserXMLDocBookTag},
     *             'id' => {@link parserIdInlineTag},
     *             'title' => {@link parserXMLDocBookTag title}),...)
     * </pre>
     */
    function setTOC($toc)
    {
        $this->toc = $toc;
    }
    
    /**
     * @param string
     */
    function setPath($path)
    {
        $this->_path = $path;
    }
    
    /**
     * @uses Converter::formatTutorialTOC() passes an array of format:
     *
     * <pre>
     * array(
     *    'tagname' => string name of tag,
     *    'link' => {@link tutorialLink} to the tutorial,
     *    'id' => converter specific tutorial ID from {@link Converter::getTutorialId()}
     *    'title' => title of the tutorial)
     * </pre>
     *
     * and returns the results as the table of contents
     * @uses Converter::getTutorialId() retrieve the tutorial ID for
     * @param Converter
     */
    function Convert(&$c)
    {
        $newtoc = array();
        foreach($this->toc as $i => $toc)
        {
            if (isset($toc['title']))
            $toc['tag']->setTitle($toc['title']);
            else
            $toc['tag']->setTitle(new parserStringWithInlineTags);
            $newtoc[$i]['tagname'] = $toc['tag']->name;
            $l = new tutorialLink;
            if (!isset($toc['title'])) $title = 'section '.$toc['id']->id;
            else
            $title = $toc['title']->Convert($c);
            $l->addLink($toc['id']->id,$this->_path,basename($this->_path),$toc['id']->package, $toc['id']->subpackage, strip_tags($title));
            $newtoc[$i]['link'] = $c->returnSee($l);
            $newtoc[$i]['id'] = $c->getTutorialId($toc['id']->package, $toc['id']->subpackage, basename($this->_path), $toc['id']->id, $toc['id']->category);
            $newtoc[$i]['title'] = $title;
        }
        return $c->formatTutorialTOC($newtoc);
    }
}
?>