<?php
//
// +------------------------------------------------------------------------+
// | phpDocumentor                                                          |
// +------------------------------------------------------------------------+
// | Copyright (c) 2000-2003 Joshua Eichorn, Gregory Beaver                 |
// | Email         jeichorn@phpdoc.org, cellog@phpdoc.org                   |
// | Web           http://www.phpdoc.org                                    |
// | Mirror        http://phpdocu.sourceforge.net/                          |
// | PEAR          http://pear.php.net/package-info.php?pacid=137           |
// +------------------------------------------------------------------------+
// | This source file is subject to version 3.00 of the PHP License,        |
// | that is available at http://www.php.net/license/3_0.txt.               |
// | If you did not receive a copy of the PHP license and are unable to     |
// | obtain it through the world-wide-web, please send a note to            |
// | license@php.net so we can mail you a copy immediately.                 |
// +------------------------------------------------------------------------+
//

/**
 * All abstract representations of DocBlock tags are defined
 * by the classes in this file
 * @package phpDocumentor
 * @subpackage DocBlockTags
 * @since separate file since version 1.2
 */
/**
 * used to represent standard tags like @access, etc.
 * This class is aware of inline tags, and will automatically handle them
 * using inherited functions
 * @package phpDocumentor
 * @subpackage DocBlockTags
 * @author Greg Beaver <cellog@users.sourceforge.net>
 * @since 1.0rc1
 * @version $Id: DocBlockTags.inc,v 1.22.2.5 2003/06/03 17:13:11 CelloG Exp $
 */
class parserTag extends parserStringWithInlineTags
{
    /**
     * Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
     * always '_tag'
     * @var string
     */
    var $type = '_tag';
    /**
     * tag name (see, access, etc.)
     * @var string
     */
    var $keyword = '';
    
    /**
     * Set up the tag
     *
     * {@source}
     * @param string $keyword tag name
     * @param parserStringWithInlineTags $value
     * @param boolean whether to parse the $value for html tags
     */
    function parserTag($keyword, $value, $noparse = false)
    {
        $this->keyword = $keyword;
        if (!$noparse)
        {
            $parser = new parserDescParser;
            $parser->subscribe('*',$this);
            $parser->parse($value->value,true,'parserstringwithinlinetags');
        } else $this->value = $value;
    }
    
    /**
     * @param Converter
     * @see Converter
     */
    function Convert(&$converter)
    {
        if (is_array($this->value))
        {
            $result = '';
            foreach($this->value as $val)
            {
                // this is only true if we processed the description in
                // the constructor
                if (get_class($val) == 'parserstringwithinlinetags')
                $result .= $converter->EncloseParagraph($val->Convert($converter));
                else
                $result .= $val->Convert($converter);
            }
            return $result;
        } else
        {
            return $this->value->Convert($converter);
        }
    }
    
    /**
     * Called by the {@link parserDescParser} when processing a description.
     * @param integer not used
     * @param array array of {@link parserStringWithInlineTags} representing
     *              paragraphs in the tag description
     * @see parserTag::parserTag()
     */
    function HandleEvent($a,$desc)
    {
        $this->value = $desc;
    }
    
    /**
     * @return string returns the text minus any inline tags
     * @see parserStringWithInlineTags::getString()
     */
    function getString()
    {
        if (is_array($this->value))
        {
            $result = '';
            foreach($this->value as $val)
            {
                $result .= $val->getString();
            }
            return $result;
        } else return $this->value->getString();
    }
}

/**
 * This class represents the @name tag
 * @tutorial tags.name.pkg
 * @package phpDocumentor
 * @subpackage DocBlockTags
 */
class parserNameTag extends parserTag
{
    /**
     * tag name
     * @var string
     */
    var $keyword = 'name';
    
    /**
     * @param string not used
     * @param string name
     */
    function parserNameTag($name, $value)
    {
        $this->value = $value;
    }
    
    /**
     * @see parserStringWithInlineTags::Convert()
     * @param Converter
     * @return string converted value of the tag
     */
    function Convert(&$c)
    {
        return $this->value;
    }
}

/**
 * This class represents the @access tag
 * @tutorial tags.access.pkg
 * @package phpDocumentor
 * @subpackage DocBlockTags
 */
class parserAccessTag extends parserTag
{
    /**
     * tag name
     * @var string
     */
    var $keyword = 'access';
    
    /**
     * set to true if the returned tag has a value type of private, protected
     * or public, false otherwise
     * @var boolean
     */
    var $isvalid = false;
    
    /**
     * checks $value to make sure it is private, protected or public, otherwise
     * it's not a valid @access tag
     * @see $isvalid
     * @param parserStringWithInlineTags $value
     */
    function parserAccessTag($value)
    {
        if (!is_string($value))
        {
            if (is_object($value))
            {
                if (method_exists($value,'getstring'))
                {
                    $value = $value->getString();
                }
            }
        }
        switch(trim($value))
        {
            case 'private' :
            case 'public' :
            case 'protected' :
                $this->value = $value;
                $this->isvalid = true;
            break;
            default :
            addError(PDERROR_ACCESS_WRONG_PARAM,$value);
                $this->value = 'public';
            break;
        }
    }
    
    /**
     * @see parserStringWithInlineTags::Convert()
     * @param Converter
     * @return string converted value of the tag
     */
    function Convert(&$converter)
    {
        return $this->value;
    }
    
    /**
     * No inline tags are possible, returns 'public', 'protected' or 'private'
     * @return string returns the text minus any inline tags
     */
    function getString()
    {
        return $this->value;
    }
}

/**
 * represents "@return"
 * @tutorial tags.return.pkg
 * @package phpDocumentor
 * @subpackage DocBlockTags
 * @author Greg Beaver <cellog@users.sourceforge.net>
 * @since 1.0rc1
 * @version $Id: DocBlockTags.inc,v 1.22.2.5 2003/06/03 17:13:11 CelloG Exp $
 */
class parserReturnTag extends parserTag
{
    /**
     * always 'return'
     * @var string
     */
    var $keyword = 'return';
    /**
     * the type a function returns
     */
    var $returnType = 'void';
    
    /**
     * contains a link to the documentation for a class passed as a type in @return, @var or @param
     *
     * Example:
     *
     * <code>
     * class myclass
     * {
     * ...
     * }
     * /** @return myclass blahblahblah
     * ...
     * </code>
     *
     * In this case, $converted_returnType will contain a link to myclass instead of the string 'myclass'
     * @var mixed either the same as $returnType or a link to the docs for a class
     * @see $returnType
     */
    var $converted_returnType = false;
    
    /**
     * @param string
     * @param parserStringWithInlineTags
     */
    function parserReturnTag($returnType, $value)
    {
        $this->returnType = $returnType;
        parent::parserTag('return',$value);
    }
    
    /**
     * sets up $converted_returnType
     * @see parserStringWithInlineTags::Convert(), $converted_returnType
     * @param Converter
     * @return string converted value of the tag
     */
    function Convert(&$converter)
    {
        $my_types = '';
        if (strpos($this->returnType,'|'))
        {
            $types = explode('|',$this->returnType);
            foreach($types as $returntype)
            {
                $a = $converter->getLink($returntype);
                if (is_object($a) && get_class($a) == 'classlink')
                {
                    if (!empty($my_types)) $my_types .= '|';
                    $my_types .= $converter->returnSee($a,$converter->type_adjust($returntype));
                } else
                {
                    if (!empty($my_types)) $my_types .= '|';
                    $my_types .= $converter->type_adjust($returntype);
                }
            }
            $this->converted_returnType = $my_types;
        } else
        {
            $a = $converter->getLink($this->returnType);
            if (is_object($a) && get_class($a) == 'classlink')
            {
                $this->converted_returnType = $converter->returnSee($a,$converter->type_adjust($this->returnType));
            } else
            {
                $this->converted_returnType = $converter->type_adjust($this->returnType);
            }
        }
        return parserTag::Convert($converter);
    }
}

/**
 * represents "@var"
 * @tutorial tags.var.pkg
 * @package phpDocumentor
 * @subpackage DocBlockTags
 * @author Greg Beaver <cellog@users.sourceforge.net>
 * @since 1.0rc1
 * @version $Id: DocBlockTags.inc,v 1.22.2.5 2003/06/03 17:13:11 CelloG Exp $
 */
class parserVarTag extends parserReturnTag
{
    /**
     * always 'var'
     * @var string
     */
    var $keyword = 'var';
    /**
     * the type a var has
     * @var string
     */
    var $returnType = 'mixed';
}

/**
 * Represents "@param"
 * @tutorial tags.param.pkg
 * @package phpDocumentor
 * @subpackage DocBlockTags
 */
class parserParamTag extends parserVarTag
{
    /**
     * always 'param'
     * @var string
     */
    var $keyword = 'param';
}

/**
 * Represents "@staticvar"
 * @tutorial tags.staticvar.pkg
 * @package phpDocumentor
 * @subpackage DocBlockTags
 */
class parserStaticvarTag extends parserParamTag
{
    /**
     * always 'staticvar'
     * @var string
     */
    var $keyword = 'staticvar';
}

/**
 * represents "@link"
 * @tutorial tags.link.pkg
 * @package phpDocumentor
 * @subpackage DocBlockTags
 * @author Greg Beaver <cellog@users.sourceforge.net>
 * @since 1.0rc1
 * @version $Id: DocBlockTags.inc,v 1.22.2.5 2003/06/03 17:13:11 CelloG Exp $
 */
class parserLinkTag extends parserTag
{
    /**
     * always 'link'
     * @var string
     */
    var $keyword = 'link';
    
    /**
     * URL to link to
     * @param string $link
     */
    function parserLinkTag($link)
    {
        $start = $val = $link->getString();
        if (strpos($val,' '))
        {
            $start = array_shift($val = explode(' ',$val));
            $val = join($val, ' ');
        }
        $a = new parserLinkInlineTag($start,$val);
        $b = new parserStringWithInlineTags;
        $b->add($a);
        $this->value = $b;
    }
}

/**
 * represents "@see"
 * @tutorial tags.see.pkg
 * @package phpDocumentor
 * @subpackage DocBlockTags
 * @author Greg Beaver <cellog@users.sourceforge.net>
 * @since 1.0rc1
 * @version $Id: DocBlockTags.inc,v 1.22.2.5 2003/06/03 17:13:11 CelloG Exp $
 */
class parserSeeTag extends parserLinkTag
{
    /**
     * always 'see'
     * @var string
     */
    var $keyword = 'see';
    
    /**
     * @param string element to link to
     */
    function parserSeeTag($name)
    {
        parserTag::parserTag($this->keyword,$name,true);
    }

    /**
     * @param Converter
     * @see parserStringWithInlineTags::Convert()
     */
    function Convert(&$converter)
    {
        if ($this->value->hasInlineTag())
        {
            addErrorDie(PDERROR_INLINETAG_IN_SEE);
        }
        $a = $converter->getLink(trim($this->value->Convert($converter)));
        if (is_string($a))
        {
            // feature 564991
            if (strpos($a,'://'))
            {
                // php function
                return $converter->returnLink($a,str_replace('PHP_MANUAL#','',$this->value->Convert($converter)));
            }
            return $a;
        }
        if (is_object($a)) return $converter->returnSee($a);
        // getLink parsed a comma-delimited list of linked thingies, add the commas back in
        if (is_array($a))
        {
            $b = '';
            foreach($a as $i => $bub)
            {
                if (!empty($b)) $b .= ', ';
                if (is_string($a[$i])) $b .= $a[$i];
                if (is_object($a[$i])) $b .= $converter->returnSee($a[$i]);
            }
            return $b;
        }
        return false;
    }
}

/**
 * represents "@license"
 *
 * Link to a license, instead of including lines and lines of license information
 * in every file
 * @tutorial tags.license.pkg
 * @package phpDocumentor
 * @subpackage DocBlockTags
 */
class parserLicenseTag extends parserLinkTag
{
    /**
     * always 'license'
     * @var string
     */
    var $keyword = 'license';
    
    /**
     * URL to link to
     * @param string $link
     */
    function parserLicenseTag($name, $link)
    {
        $a = explode(' ',$link->getString());
        $url = array_shift($a);
        $license = join($a,' ');
        if (empty($license)) $license = $url;
        $a = new parserLinkInlineTag($url, $license);
        $b = new parserStringWithInlineTags;
        $b->add($a);
        $this->value = $b;
    }
}

/**
 * represents "@uses"
 *
 * This is exactly like @see except that the element used has a @useby link to this element added to its docblock
 * @package phpDocumentor
 * @subpackage DocBlockTags
 * @tutorial tags.uses.pkg
 * @since 1.2
 */
class parserUsesTag extends parserSeeTag
{
    /**
     * Always "uses"
     * @var string
     */
    var $keyword = 'uses';
    /** @access private */
    var $_description;
    
    /**
     * @param string element to link to
     * @param parserStringWithInlineTags description of how the element is used
     */
    function parserUsesTag($seeel, $description)
    {
        parent::parserSeeTag($seeel);
        $this->_description = $description;
    }
    
    /**
     * Return a link to documentation for other element, and description of how
     * it is used
     *
     * Works exactly like {@link parent::Convert()} except that it also includes
     * a description of how the element used is used.
     * @return string
     * @param Converter
     */
    function Convert(&$c)
    {
        $val = $this->value;
        $see = parent::Convert($c);
        $this->value = $this->_description;
        $see .= ' - '.parserTag::Convert($c);
        $this->value = $val;
        return $see;
    }
    
    /**
     * Get the text of the link to the element that is being used
     * @return string
     * @access private
     */
    function getSeeElement()
    {
        return $this->value->getString();
    }
    
    /**
     * Get the description of how the element used is being used.
     * @return parserStringWithInlineTags
     */
    function getDescription()
    {
        return $this->_description;
    }
}

/**
 * This is a virtual tag, it is created by @uses to cross-reference the used element
 *
 * This is exactly like @uses.
 * @package phpDocumentor
 * @subpackage DocBlockTags
 * @since 1.2
 */
class parserUsedByTag extends parserUsesTag
{
    /**
     * Always "usedby"
     * @var string
     */
    var $keyword = 'usedby';
    /** @access private */
    var $_link;
    
    /**
     * @param abstractLink link of element that uses this element
     * @param string description of how the element is used
     */
    function parserUsedByTag($link, $description)
    {
        $this->value = $description;
        $this->_link = $link;
    }
    
    /**
     * @return string
     * @param Converter
     */
    function Convert(&$c)
    {
        $see = $c->returnSee($this->_link);
        $see .= ' - '.parserTag::Convert($c);
        return $see;
    }
}

/**
 * represents "@tutorial"
 *
 * This is exactly like @see except that it only links to tutorials
 * @tutorial phpDocumentor/tutorials.pkg
 * @tutorial tags.tutorial.pkg
 * @package phpDocumentor
 * @subpackage DocBlockTags
 * @since 1.2
 */
class parserTutorialTag extends parserSeeTag
{
    /**
     * Always "tutorial"
     * @var string
     */
    var $keyword = 'tutorial';
    /**
     * @param Converter
     * @see parserStringWithInlineTags::Convert()
     */
    function Convert(&$converter)
    {
        $a = $converter->getTutorialLink(trim($this->value->Convert($converter)));
        if (is_string($a))
        {
            return $a;
        }
        if (is_object($a)) return $converter->returnSee($a);
        // getLink parsed a comma-delimited list of linked thingies, add the commas back in
        if (is_array($a))
        {
            $b = '';
            foreach($a as $i => $bub)
            {
                if (!empty($b)) $b .= ', ';
                if (is_string($a[$i])) $b .= $a[$i];
                if (is_object($a[$i])) $b .= $converter->returnSee($a[$i]);
            }
            return $b;
        }
        return false;
    }
}

/**
 * represents "@filesource"
 * 
 * Use this to create a link to a highlighted phpxref-style source file listing
 * @package phpDocumentor
 * @subpackage DocBlockTags
 * @tutorial tags.filesource.pkg
 */
class parserFileSourceTag extends parserTag
{
    /**
     * Always "filesource"
     * @var string
     */
    var $keyword = 'filesource';
    /** @var array */
    var $source;
    /** @var string */
    var $path;
    
    /**
     * Set {@link $source} to $value, and set up path
     * @param string
     * @param array output from {@link phpDocumentorTWordParser::getFileSource()}
     */
    function parserFileSourceTag($filepath, $value)
    {
        parent::parserTag($this->keyword, '');
        $this->path = $filepath;
        $this->source = $value;
    }

    /**
     * Return a link to the highlighted source and generate the source
     * @uses ConvertSource() generate source code and write it out
     * @return string output from {@link getSourceLink()}
     * @param Converter
     */
    function Convert(&$c)
    {
        $this->ConvertSource($c);
        return $this->getSourceLink($c);
    }

    /**
     * @param Converter
     * @uses phpDocumentor_HighlightParser highlights source code
     * @uses writeSource()
     */
    function ConvertSource(&$c)
    {
        $parser = new phpDocumentor_HighlightParser;
        $return = '';
        $return = $parser->parse($this->source,$c, false, false, false, $this->path);
        $this->writeSource($c, $return);
    }

    /**
     * @param Converter
     * @param string highlighted source code
     * @uses Converter::writeSource() export highlighted file source
     */
    function writeSource(&$c, $source)
    {
        $c->writeSource($this->path, $source);
    }

    /**
     * @uses Converter::getSourceLink()
     * @param Converter
     * @return output from getSourceLink()
     */
    function getSourceLink(&$c)
    {
        return $c->getSourceLink($this->path);
    }
}

/**
 * represents "@example"
 * 
 * The example tag 
 * @package phpDocumentor
 * @subpackage DocBlockTags
 * @tutorial tags.example.pkg
 */
class parserExampleTag extends parserFileSourceTag
{
    /**
     * always "example"
     * @var string
     */
    var $keyword = 'example';
    /**
     * Reads and parses the example file indicated
     *
     * The example tag takes one parameter: the full path to a php file that
     * should be parsed and included as an example.
     * @uses phpDocumentorTWordParser::getFileSource() uses to parse an example
     *       and retrieve all tokens by line number
     * @param string
     * @param parserStringWithInlineTags
     */
    function parserExampleTag($tag, $value)
    {
        global $_phpDocumentor_phpfile_exts, $_phpDocumentor_setting;
        parent::parserTag($tag, $value);
        $exts = $_phpDocumentor_phpfile_exts;
        $path = false;
        $filename = $value->getString();
        $state = 0;
        do
        {
            $ext = array_pop($exts);
            $test = 0; // state variable
            if (strpos($filename,'.'.$ext) === false)
            {
                $path = false;
            } else
            {
                $original_path = substr($filename,0,strpos($filename,'.'.$ext) + strlen($ext) + 1);

                // Is the path absolute? (i.e. does it start like an absolute path?)
                if ((!PHPDOCUMENTOR_WINDOWS && $original_path{0} == '/') ||
                    (PHPDOCUMENTOR_WINDOWS && preg_match('`^([A-Za-z]:){0,1}\\\`', $original_path)))
                {
                    // Yes, just use the path as provided
                	$path = $original_path;
                }
                else
                {
                    // no, use the example directory, if passed in
                    if (isset($_phpDocumentor_setting['examplesdir']) && ($state === 0))
                    {
                        $dir = realpath($_phpDocumentor_setting['examplesdir']);
                    } else
                    {
                        // No example directory, see if there is an 'examples' subdirectory
                        if (is_dir(PHPDOCUMENTOR_BASE . PATH_DELIMITER . 'examples') && ($state == 1))
                        {
                            $dir = PHPDOCUMENTOR_BASE . PATH_DELIMITER . 'examples';
                        } else
                        {
                            // just use the base directory
                            $dir = PHPDOCUMENTOR_BASE;
                        }
                    }
                	$path = realpath($dir. PATH_DELIMITER . $original_path);
                    $state++; // make sure we cycle through the possibilities
                }
            }
        } while (!$path || (!@file_exists($path) && is_file($path) && count($exts)));
        $value = substr($filename,strlen($original_path));
        if (!$value) $value = 'example';
        $this->_title = $value;
        $this->path = str_replace(array(':', DIRECTORY_SEPARATOR,'/'), array('_', '_', '_'), $original_path);
        if (!@file_exists($path))
        {
            addWarning(PDERROR_EXAMPLE_NOT_FOUND,$path);
            $this->path = false;
        } else
        {
            $f = @fopen($path,'r');
            if ($f)
            {
                $example = fread($f,filesize($path));
                $obj = new phpDocumentorTWordParser;
                $obj->setup($example);
                $this->source = $obj->getFileSource();
                unset($obj);
            }
        }
    }
    
    /**
     * @param Converter $c
     * @param string parsed example source
     * @uses Converter::writeExample() writes final example out
     * @access private
     */
    function writeSource(&$c, $source)
    {
        if ($this->path)
        $c->writeExample($this->_title, $this->path, $source);
    }

    /**
     * Retrieve a converter-specific link to the example
     *
     * @param Converter
     * @uses Converter::getExampleLink() retrieve the link to the example
     */
    function getSourceLink(&$c)
    {
        if (!$this->path) return $this->_title;
        return $c->getExampleLink($this->path, $this->_title);
    }
}

?>